import threading
import time
from graphics_and_games_klassen import Rechteck, Figur  

from intern.zeichenfenster import Zeichenfenster

class Kiste(Rechteck):
    """
    Kiste beim Erzeuger-Verbraucher-Problem
    """
    def __init__(self, nummer):
        """
        Konstruktur der Klasse Kiste
        :param nummer: Nummer der Kiste
     */
     """
        super().__init__(farbe="braun")
        self.nummer = nummer # laufende Nummer der Kiste

    def ImSpeicherDarstellen(self):
        """
        Stellt die Kiste auf dem Speicherplatz dar.
        """
        self.PositionSetzen(375, 200)
        self.SichtbarkeitSetzen(True)

    def SpeicherdarstellungVerbergen(self):
        """
        Verbirgt die Kiste auf dem Speicherplatz.
        """
        self.SichtbarkeitSetzen(False)

    def KistennummerGeben(self):
        """
        Meldet die Nummer der Kiste.
        :return: Kistennummer
        """
        return self.nummer

class Speicher(Rechteck):
    """
    Speicherplatz für Kisten beim Erzeuger-Verbraucher-Problem
    """
    def __init__(self):
        """
        Konstruktor der Klasse Speicher
        """
        super().__init__(375, 250, farbe="grau", breite=50, hoehe=10)
        self.frei = True # Gibt an, ob Platz frei ist
        self.lock = threading.Lock() # Schutz für den Speicherplatz
        self.kiste = None # Die eventuell vorhandene Kiste

    def Ablegen(self, kiste):
        """
        Legt eine Kiste auf den Speicherplatz ab.
        :param kiste: Die abzulegende Kiste
        :return: True, wenn die Kiste abgelegt werden konnte, sonst False
        """

        with self.lock:

            if self.frei:
                self.frei = False
                self.kiste = kiste
                kiste.ImSpeicherDarstellen()
                return True
            return False

    def Holen(self):
        """
        Holt eine Kiste vom Speicherplatz.
        :return: Die geholte Kiste oder None, wenn keine Kiste vorhanden ist
        """
        with self.lock:

            if not self.frei:
                self.frei = True
                kiste = self.kiste
                kiste.SpeicherdarstellungVerbergen()
                self.kiste = None
                return kiste
            return None

class Erzeuger(threading.Thread):
    """
    Erzeuger beim Erzeuger-Verbraucher-Problem
    """
    def __init__(self, zeit, speicher):
        """
        Konstruktor der Klasse Erzeuger
        :param zeit: Zeit, die der Erzeuger für die Produktion einer Kiste benötigt
        :param speicher: Speicherplatz für die Kisten
        """
        super().__init__()
        self.zeit = zeit
        self.speicher = speicher
        self.nummer = 0
        self.erzeugerSymbol = ErzeugerSymbol()

    def run(self):
        """
        Hauptmethode des Threads
        """
        while True:
            kiste = self.Produzieren()
            abgelegt = False
            self.BeladenHinDarstellen()
            abgelegt = self.speicher.Ablegen(kiste)
            while not abgelegt:
                self.BeladenZurueckDarstellen()
                self.BeladenHinDarstellen()
                abgelegt = self.speicher.Ablegen(kiste)
            self.LeerZurueckDarstellen()

    def Produzieren(self):
        """
        Produziert eine Kiste.
        """
        self.nummer += 1
        kiste = Kiste(self.nummer)

        kiste.PositionSetzen(125, 200)
        for zaehler in range(10):
            kiste.GroesseSetzen(5 * (zaehler + 1), 50)
            time.sleep(self.zeit / 1000)

        kiste.SichtbarkeitSetzen(False)
        return kiste

    def BeladenHinDarstellen(self):
        """
        Stellt die Bewegung des Erzeugers mit der Kiste beim Beladen dar. 
        """
        self.erzeugerSymbol.FigurMitKisteFestlegen()
        for i in range(10):
            self.erzeugerSymbol.BeladenHin(i / 9.0)
            time.sleep(self.zeit / 1000)

    def BeladenZurueckDarstellen(self):
        """
        Stellt die Bewegung des Erzeugers mit der Kiste zurück dar. 
        """
        self.erzeugerSymbol.FigurMitKisteFestlegen()
        for i in range(10):
            self.erzeugerSymbol.BeladenZurueck(i / 9.0)
            time.sleep(self.zeit / 1000)

    def LeerZurueckDarstellen(self):
        """
        Stellt die Bewegung des Erzeugers ohne Kiste zurück dar. 
        """
        self.erzeugerSymbol.FigurOhneKisteFestlegen()
        for i in range(10):
            self.erzeugerSymbol.LeerZurueck(i / 9.0)
            time.sleep(self.zeit / 1000)

class ErzeugerSymbol(Figur):
    """
    Symbol für den Erzeuger
    """
    def __init__(self):
        super().__init__(x=275, y=225)
        self.FigurOhneKisteFestlegen()
        self.GroesseSetzen(100)
        self.PositionSetzen(275, 225)
        self.LeerZurueck(1)

    def BeladenHin(self, anteil):
        """
        Schwenkt den beladenen Erzeugerarm Richtung Ablage
        :param anteil: Winkelanzeil des Schwenkbereichs
        """
        winkel = int(180 - 150 * anteil) # Anteil an 180° in Winkel umrechnen und Winkel setzen
        self.WinkelSetzen(winkel)

    def BeladenZurueck(self, anteil):
        """
        Schwenkt den beladenen Erzeugerarm Richtung Ausgangslage
        :param anteil: Winkelanzeil des Schwenkbereichs    
        """
        winkel = int(150 * anteil + 30) # Anteil an 180° in Winkel umrechnen und Winkel setzen
        self.WinkelSetzen(winkel)

    def LeerZurueck(self, anteil):
        """
        Schwenkt den beladenen Erzeugerarm Richtung Ausgangslage
        :param anteil: Winkelanzeil des Schwenkbereichs
        """
        winkel = int(150 * anteil + 30) # Anteil an 180° in Winkel umrechnen und Winkel setzen
        self.WinkelSetzen(winkel)

    def FigurMitKisteFestlegen(self):
        """
        Erzeugt den Ableger mit einer Kiste
        """
        self.EigeneFigurLoeschen()
        self.FigurteilFestlegenRechteck(0, -10, 100, 20, "schwarz")
        self.FigurteilFestlegenEllipse(-10, -10, 20, 20, "schwarz")
        self.FigurteilFestlegenRechteck(100, -25, 50, 50, "braun")

    def FigurOhneKisteFestlegen(self):
        """
        Erzeugt den Ableger ohe Kiste
        """
        self.EigeneFigurLoeschen() # Bisherige Darstellung löschen
        # Figur ohne Kiste darstellen
        self.FigurteilFestlegenRechteck(0, -10, 100, 20, "schwarz")
        self.FigurteilFestlegenEllipse(-10, -10, 20, 20, "schwarz")

class Verbraucher(threading.Thread):
    """
    Verbraucher von Kisten.
    """
    def __init__(self, zeit, speicher):
        """
        Konstruktor für Objekte der Klasse Verbraucher
        :param zeit: Ablagezeit für die Kisten
        :param speicher: der Speicher, aus dem geholt werden soll
        """
        super().__init__()
        self.zeit = zeit #  Die Produktionszeit pro Kiste
        self.speicher = speicher # Der Zwischenspeicher
        self.verbraucherSymbol = VerbraucherSymbol() # Die Darstellung des Verbrauchers

    def run(self):
        """
        Die Arbeitsmethode des Threads.
        """
        while True:
            self.LeerHinDarstellen()
            kiste = self.speicher.Holen()
            while kiste is None:
                self.LeerZurueckDarstellen()
                self.LeerHinDarstellen()
                kiste = self.speicher.Holen()
            self.BeladenZurueckDarstellen()
            self.Einlagern(kiste)

    def LeerHinDarstellen(self):
        """ 
        Stellt den Verbraucher ohne Kiste auf dem Weg zum Abholplatz dar.
        """
        self.verbraucherSymbol.FigurOhneKisteFestlegen()
        for i in range(10):
            self.verbraucherSymbol.LeerHin(i / 9)
            time.sleep(self.zeit / 1000)

    def LeerZurueckDarstellen(self):
        """
        Stellt den Verbraucher ohne Kiste auf dem Weg vom Abholplatz zurück dar.
        """
        self.verbraucherSymbol.FigurOhneKisteFestlegen()
        for i in range(10):
            self.verbraucherSymbol.LeerZurueck(i / 9)
            time.sleep(self.zeit / 1000)

    def BeladenZurueckDarstellen(self):
        """
        Stellt den Verbraucher mit Kiste auf dem Weg vom Abholplatz zurück dar.
        """
        self.verbraucherSymbol.FigurMitKisteFestlegen()
        for i in range(10):
            self.verbraucherSymbol.BeladenZurueck(i / 9)
            time.sleep(self.zeit / 1000)
        self.verbraucherSymbol.FigurOhneKisteFestlegen()

    def Einlagern(self, kiste):
        """
        Lagert Kisten in der gegebenen Zeit.
        :param kisteNeu: die zu lagernde Kiste
        """
        for zaehler in range(10):
            kiste.SichtbarkeitSetzen(True)
            kiste.PositionSetzen(625 + 5 * (zaehler + 1), 200)
            kiste.GroesseSetzen(50 - 5 * (zaehler + 1), 50)
            time.sleep(self.zeit / 1000)

class VerbraucherSymbol(Figur):
    """
    Darstellung des Verbrauchers
    """
    def __init__(self):
        """
        Konstruktor der Klasse VerbraucherSymbol
        """
        super().__init__(x=525, y=225)
        self.FigurOhneKisteFestlegen()
        self.GroesseSetzen(100)
        self.PositionSetzen(525, 225)
        self.LeerZurueck(1)

    def LeerHin(self, anteil):
        """
        Schwenkt den leeren Verbraucherarm Richtung Ablage
        :param anteil: Winkelanzeil des Schwenkbereichs
        """
        winkel = int(160 * anteil)
        self.WinkelSetzen(winkel)

    def LeerZurueck(self, anteil):
        """
        Schwenkt den leeren Verbraucherarm Richtung Einlagerung
        :param anteil: Winkelanzeil des Schwenkbereichs
        """
        winkel = int(160 - 160 * anteil)
        self.WinkelSetzen(winkel)

    def BeladenZurueck(self, anteil):
        """
        Schwenkt den beladenen Verbraucherarm Richtung Einlagerung
        :param anteil: Winkelanzeil des Schwenkbereichs
        """
        winkel = int(160 - 160 * anteil)
        self.WinkelSetzen(winkel)

    def FigurMitKisteFestlegen(self):
        """
        Erzeugt den Konsumenten mit einer Kiste
        """
        self.EigeneFigurLoeschen()
        self.FigurteilFestlegenRechteck(0, -10, 100, 20, "schwarz")
        self.FigurteilFestlegenEllipse(-10, -10, 20, 20, "schwarz")
        self.FigurteilFestlegenRechteck(100, -25, 50, 50, "braun")

    def FigurOhneKisteFestlegen(self):
        """
        Erzeugt den Konsumenten ohne Kiste
        """
        self.EigeneFigurLoeschen()
        self.FigurteilFestlegenRechteck(0, -10, 100, 20, "schwarz")
        self.FigurteilFestlegenEllipse(-10, -10, 20, 20, "schwarz")


if __name__ == "__main__":
    speicher = Speicher()
    erzeuger = Erzeuger(50, speicher)
    verbraucher = Verbraucher(250, speicher)
    erzeuger.start()
    verbraucher.start()

    Zeichenfenster().run()