import threading
import time
import random
from threading import Condition, Thread

class Erzeuger(Thread):
    """
    Erzeuger von Kisten.
    """
    def __init__(self, zeitNeu, speicherNeu):
        """
        Konstruktor für Objekte der Klasse Erzeuger.
        
        :param zeitNeu: Produktionszeit für die Kisten in Millisekunden
        :param speicherNeu: der Speicher, in dem abgelegt werden soll
        """
        super().__init__()
        self.zeit = zeitNeu
        self.speicher = speicherNeu
        self.nummer = 0

    def run(self):
        """
        Die Arbeitsmethode des Threads.
        Führt die Erzeugung von Kisten kontinuierlich aus.
        """
        while True:
            kNeu = self.Produzieren()
            print(f"Ablegeversuch von Kiste {kNeu.KistennummerGeben()}")
            self.speicher.Ablegen(kNeu)
            print(f"Kiste {kNeu.KistennummerGeben()} abgelegt.")

    def Produzieren(self):
        """
        Produziert Kisten in der gegebenen Zeit.
        
        :return: die neue Kiste
        """
        akt = time.time()
        ende = akt + self.zeit / 1000  # Umrechnung von Millisekunden in Sekunden
        while akt < ende:
            time.sleep(max(0, ende - akt))  # Warten bis die Produktionszeit abgelaufen ist
            akt = time.time()
        self.nummer += 1
        print(f"Kiste Nummer {self.nummer} produziert.")
        return Kiste(self.nummer)

class Kiste:
    """
    Kiste beim Erzeuger-Verbraucher-Problem.
    """
    def __init__(self, nummerNeu):
        """
        Konstruktor, der die Kistennummer festlegt.
        
        :param nummerNeu: Nummer der Kiste
        """
        self.nummer = nummerNeu

    def KistennummerGeben(self):
        """
        Meldet die Nummer der Kiste.
        
        :return: Kistennummer
        """
        return self.nummer

class Speicher:
    """
    Zwischenspeicher beim Erzeuger-Verbraucher-Problem.
    """
    def __init__(self):
        """
        Initialisiert den Speicher, setzt ihn als frei und ohne Kiste.
        """
        self.frei = True
        self.kiste = None
        self.condition = Condition()

    def Ablegen(self, kisteNeu):
        """
        Ablegen der Kiste im Speicher.
        
        :param kisteNeu: die abzulegende Kiste
        """
        with self.condition:
            while not self.frei:
                self.condition.wait()  # Warten, bis der Speicher frei ist
            self.frei = False        
            self.kiste = kisteNeu
            self.condition.notify()

    def Holen(self):
        """
        Holt eine Kiste aus dem Speicher.
        
        :return: die geholte Kiste
        """
        with self.condition:
            while self.frei:
                self.condition.wait()  # Warten, bis eine Kiste verfügbar ist
            self.frei = True
            k = self.kiste
            self.kiste = None
            self.condition.notify()
            return k

class Verbraucher(Thread):
    """
    Verbraucher von Kisten.
    """
    def __init__(self, zeitNeu, speicherNeu):
        """
        Konstruktor für Objekte der Klasse Verbraucher.
        
        :param zeitNeu: Ablagezeit für die Kisten in Millisekunden
        :param speicherNeu: der Speicher, aus dem geholt werden soll
        """
        super().__init__()
        self.zeit = zeitNeu
        self.speicher = speicherNeu
        self.zufallsgenerator = random.Random()

    def run(self):
        """
        Die Arbeitsmethode des Threads.
        Holt kontinuierlich Kisten aus dem Speicher und lagert sie ein.
        """
        while True:
            kNeu = self.speicher.Holen()
            self.Einlagern(kNeu)

    def Einlagern(self, kisteNeu):
        """
        Lagert Kisten in der gegebenen Zeit ein.
        
        :param kisteNeu: die zu lagernde Kiste
        """
        akt = time.time()
        ende = akt + (self.zeit / 2 + self.zufallsgenerator.randint(0, self.zeit)) / 1000
        while akt < ende:
            time.sleep(max(0, ende - akt))  # Warten bis die Einlagerungszeit abgelaufen ist
            akt = time.time()
        print(f"Kiste Nummer {kisteNeu.KistennummerGeben()} eingelagert.")

# Beispielcode
speicher = Speicher()
erzeuger = Erzeuger(1000, speicher)
verbraucher = Verbraucher(1500, speicher)
erzeuger.start()
verbraucher.start()
