import Foundation

/**
 Simuliert ein (langsames) LCD-Display
 */
class LCDDisplay
{
    /**
     Gibt eine Zeichenkette auf das Display aus
     - parameters:
        - text: die auszugebende Zeichenkette
     */
    func TextAusgeben(text: String)
    {
        for z in text
        {
            ZeichenAusgeben(zeichen: z)
        }
    }
    
    /**
     Gibt ein einzelnes Zeichen auf das Display aus
     - parameters:
        - zeichen: das auszugebende Zeichen
     */
    private func ZeichenAusgeben(zeichen: Character)
    {
        // Simuliert ein Hardwaredisplay, welches nur zeichenweise
        // und mit begrenzter Geschwindigkeit beschrieben werden kann.
        print(zeichen, terminator: "")
        Thread.sleep(forTimeInterval: 0.01)
    }
}

/**
 Oberklasse aller Sensoren
 */
class Sensor: Thread
{
    /** Typ des Sensors */
    fileprivate var typ: String
    /** Ausgabeeinheit */
    private var anzeige: LCDDisplay

    /**
     Konstruktor für Sensoren, setzt die Ausgabeeinheit
     - parameters:
        - anzeige: die Ausgabeeinheit
     */
    init(anzeige: LCDDisplay)
    {
        self.anzeige = anzeige
        typ = "Sensor"
        super.init()
    }
    
    /**
     Meldet den aktuellen Messwert des Sensors
     - returns Messwert
     */
    func MesswertGeben() -> String
    {
        fatalError("Sensor.MesswertGeben muss überschrieben werden")
    }
    
    /**
     Hauptmethode des Threads
     */
    override func main()
    {
        while(true)
        {
            Thread.sleep(forTimeInterval: 2.0); // Zwei Sekunden Wartezeit zwischen den Messwertausgaben
            let messwert = MesswertGeben()
            let textzeile = "Sensor: " + typ + "\tMesswert: " + messwert + "\n"
            anzeige.TextAusgeben(text: textzeile)
        }
    }
}

/**
 Ein Temperatursensor
 */
class Temperatursensor: Sensor
{
    /**
     Konstruktor für Sensoren, setzt die Ausgabeeinheit
     - parameters:
        - anzeige: die Ausgabeeinheit
     */
    override init(anzeige: LCDDisplay)
    {
        super.init(anzeige: anzeige)
        typ = "Temperatur"
    }
    
    /**
     Meldet den aktuellen Messwert des Sensors
     - returns Messwert
     */
    override func MesswertGeben() -> String
    {
        return "\(Int.random(in: -10...40)) °C"
    }
}

/**
 Ein Luftdrucksensor
 */
class Luftdrucksensor: Sensor
{
    /**
     Konstruktor für Sensoren, setzt die Ausgabeeinheit
     - parameters:
        - anzeige: die Ausgabeeinheit
     */
    override init(anzeige: LCDDisplay)
    {
        super.init(anzeige: anzeige)
        typ = "Luftdruck"
    }
    
    /**
     Meldet den aktuellen Messwert des Sensors
     - returns Messwert
     */
    override func MesswertGeben() -> String
    {
        return "\(Int.random(in: 1000...1029)) hPa"
    }
}

/**
 Ein Luftfeuchtesensor
 */
class Luftfeuchtesensor: Sensor
{
    /**
     Konstruktor für Sensoren, setzt die Ausgabeeinheit
     - parameters:
        - anzeige: die Ausgabeeinheit
     */
    override init(anzeige: LCDDisplay)
    {
        super.init(anzeige: anzeige)
        typ = "Luftfeuchte"
    }
    
    /**
     Meldet den aktuellen Messwert des Sensors
     - returns Messwert
     */
    override func MesswertGeben() -> String
    {
        return "\(Int.random(in: 0...100)) %"
    }
}

/**
 Ein Windgeschwindigkeitssensor
 */
class Windgeschwindigkeitssensor: Sensor
{
    /**
     Konstruktor für Sensoren, setzt die Ausgabeeinheit
     - parameters:
        - anzeige: die Ausgabeeinheit
     */
    override init(anzeige: LCDDisplay)
    {
        super.init(anzeige: anzeige)
        typ = "Windgeschwindigkeit"
    }
    
    /**
     Meldet den aktuellen Messwert des Sensors
     - returns Messwert
     */
    override func MesswertGeben() -> String
    {
        return "\(Int.random(in: -0...200)) km/h"
    }
}

/**
 Ein Windrichtungssensor
 */
class Windrichtungssensor: Sensor
{
    /**
     Konstruktor für Sensoren, setzt die Ausgabeeinheit
     - parameters:
        - anzeige: die Ausgabeeinheit
     */
    override init(anzeige: LCDDisplay)
    {
        super.init(anzeige: anzeige)
        typ = "Windrichtung"
    }
    
    /**
     Meldet den aktuellen Messwert des Sensors
     - returns Messwert
     */
    override func MesswertGeben() -> String
    {
        return  ["N", "NO", "O", "SO", "S", "SW", "W", "NW"] [Int.random(in: 0..<8)]
    }
}

/**
 Eine Wettersation mit typischen Sensoren
 */
class Wetterstation
{
    init()
    {
        let anzeige = LCDDisplay()
        let sensoren = [Temperatursensor(anzeige: anzeige), Luftdrucksensor(anzeige: anzeige), Luftfeuchtesensor(anzeige: anzeige), Windgeschwindigkeitssensor(anzeige: anzeige), Windrichtungssensor(anzeige: anzeige)]
        for s in sensoren
        {
            s.start()
        }
    }
}

Wetterstation()
