import Foundation

/**
Kiste beim Erzeuger-Verbraucher-Problem

- author: Johannes Neumeyer
- version: 1.0
*/
class Kiste: Rechteck
{
    /** Laufende Nummer der Kiste */
    private var nummer: Int

    /**
    Legt die Kistennummer fest.
    - parameters:
        - nummerNeu: Nummer der Kiste
    */
    init(nummerNeu: Int)
    {
        nummer = nummerNeu
        super.init()
        FarbeSetzen(farbe: "braun")
    }

    /**
    Stellt die Kiste auf dem Speicherplatz dar.
    */
    func ImSpeicherDarstellen()
    {
        PositionSetzen(x: 375, y: 200)
        SichtbarkeitSetzen(sichtbar: true)
    }

    /**
    Verbirgt die Kiste auf dem Speicherplatz.
    */
    func SpeicherdarstellungVerbergen()
    {
        SichtbarkeitSetzen(sichtbar: false)
    }

    /**
    Meldet die Nummer der Kiste.
    - returns: Kistennummer
    */
    func KistennummerGeben() -> Int
    {
        return nummer
    }
}


/**
Verwaltet den Zwischenspeicher und stellt ihn dar.

- author: Johannes Neumeyer
- version: 1.0
*/
class Speicher: Rechteck
{
    /** Gibt an, ob Platz frei ist */
    private var frei: Bool
    /** Die eventuell vorhandene Kiste */
    private var kiste: Kiste?
    /** zur Steuerung der Kistendarstellung */
    private var kistendarstellung: Kiste
    /** Monitor zur Steuerung der Ablage */
    private let monitor: NSCondition

    /**
    Besetzt die Attribute vor.
    */
    override init()
    {
        frei = true
        kiste = nil
        kistendarstellung = Kiste(nummerNeu: -1)
        kistendarstellung.SichtbarkeitSetzen(sichtbar: false)
        monitor = NSCondition()
        super.init()

        GrößeSetzen(breite: 50, höhe: 10)
        FarbeSetzen(farbe: "grau")
        PositionSetzen(x: 375, y: 250)
    }

    /**
    Versucht, eine Kiste abzulegen und wartet gegebenenfalls.
    - parameters:
        - kiste: die abzulegende Kiste
    */
    func Ablegen(kisteNeu: Kiste) -> Bool
    {
        monitor.lock()
        if frei
        {
            frei = false
            kiste = kisteNeu
            kiste!.ImSpeicherDarstellen()
            monitor.unlock()
            return true
        }
        monitor.unlock()
        return false
    }

    /**
    Versucht, eine Kiste zu holen und wartet gegebenenfalls.
    - returns: die geholte Kiste
    */
    func Holen() -> Kiste?
    {
        monitor.lock()
        if !frei
        {
            frei = true
            let kNeu = kiste
            kiste!.SpeicherdarstellungVerbergen();
            kiste = nil
            monitor.unlock()
            return kNeu
        }
        monitor.unlock()
        return nil
    }
}

/**
Erzeuger von Kisten.

- author: Johannes Neumeyer
- version: 1.0
*/
class Erzeuger: Thread
{
    /** Die Produktionszeit pro Kiste in Millisekunden */
    private var zeit: Int
    /** Der Zwischenspeicher */
    private var speicher: Speicher
    /** Die Darstellung des Erzeugers **/
    private var erzeugerSymbol: ErzeugerSymbol
    /** Laufende Nummer der produzierten Kisten */
    private var nummer: Int

    /**
    Konstruktor für Objekte der Klasse Erzeuger
    - parameters:
        - zeitNeu: Produktionszeit für die Kisten
        - speicherNeu: der Speicher, in dem abgelegt werden soll
    */
    init(zeitNeu: Int, speicherNeu: Speicher)
    {
        zeit = zeitNeu
        speicher = speicherNeu
        nummer = 0
        erzeugerSymbol = ErzeugerSymbol()
    }

    /**
     * Die Arbeitsmethode des Threads.
     */
    override func main()
    {
        while true
        {
            let kNeu = Produzieren()
            var abgelegt = false

            BeladenHinDarstellen()
            abgelegt = speicher.Ablegen(kisteNeu: kNeu)

            while !abgelegt
            {
                BeladenZurückDarstellen()
                BeladenHinDarstellen()
                abgelegt = speicher.Ablegen(kisteNeu: kNeu)
            }
            LeerZurückDarstellen();
        }
    }

    /**
    Produziert Kisten in der gegebenen Zeit.
    - returns: die neue Kiste
     */
    private func Produzieren() -> Kiste
    {
        nummer = nummer + 1
        let kNeu = Kiste(nummerNeu: nummer)

        kNeu.PositionSetzen(x: 125, y: 200)

        for zähler in  0 ..< 10
        {
            kNeu.GrößeSetzen(breite: 5 * (zähler + 1), höhe: 50)

            Thread.sleep(forTimeInterval: Double(zeit) / 10000.0)

        }

        kNeu.SichtbarkeitSetzen(sichtbar: false)
        return kNeu
    }

    /**
    Stellt den Erzeuger mit Kiste auf dem Weg zum Ablegeplatz dar.
     */
    private func BeladenHinDarstellen()
    {
        erzeugerSymbol.FigurMitKisteFestlegen();
        for i in 0 ..< 10
        {
            erzeugerSymbol.BeladenHin(anteil: Double(i) / 9)
            Thread.sleep(forTimeInterval: Double(zeit) / 10000.0)
        }
    }

    /**
    Stellt den Erzeuger mit Kiste auf dem Weg vom Ablegeplatz zurück dar.
    */
    private func BeladenZurückDarstellen()
    {
        erzeugerSymbol.FigurMitKisteFestlegen()
        for i in 0 ..< 10
        {
            erzeugerSymbol.BeladenZurück(anteil: Double(i) / 9)
            Thread.sleep(forTimeInterval: Double(zeit) / 10000.0)
        }
    }

    /**
    Stellt den Erzeuger ohne Kiste auf dem Weg vom Ablegeplatz zurück dar.
     */
    private func LeerZurückDarstellen()
    {
        erzeugerSymbol.FigurOhneKisteFestlegen();
        for i in 0 ..< 10
        {
            erzeugerSymbol.LeerZurück(anteil: Double(i) / 9)
            Thread.sleep(forTimeInterval: Double(zeit) / 10000.0)
        }
    }

    /**
    Darstellung des Erzeugers
    
    - author: Johannes Neumeyer
    - version 1.0
    */
    class ErzeugerSymbol: Figur
    {
        /**
        Erzeugt das Symbol in Startstellung
        */
        override init()
        {
            super.init()
            FigurOhneKisteFestlegen()
            GrößeSetzen(größe: 100)
            PositionSetzen(x: 275, y: 225)
            LeerZurück(anteil: 1)
        }

        /**
        Schwenkt den beladenen Erzeugerarm Richtung Ablage
         - parameters:
             - anteil: Winkelanzeil des Schwenkbereichs
        */
        func BeladenHin(anteil: Double)
        {
            // Anteil an 180° in Winkel umrechnen und Winkel setzen
            let winkel = Int(180 - 150 * anteil)
            WinkelSetzen(winkel: winkel)
        }

        /**
         Schwenkt den beladenen Erzeugerarm Richtung Ausgangslage
         - parameters:
             - anteil: Winkelanzeil des Schwenkbereichs
         */
        func BeladenZurück(anteil: Double)
        {
            // Anteil an 180° in Winkel umrechnen und Winkel setzen
            let winkel = Int(150 * anteil + 30)
            WinkelSetzen(winkel: winkel)
        }

        /**
         * Schwenkt den beladenen Erzeugerarm Richtung Ausgangslage
         - parameters:
             - anteil: Winkelanzeil des Schwenkbereichs
         */
        func LeerZurück(anteil: Double)
        {
            // Anteil an 180° in Winkel umrechnen und Winkel setzen
            let winkel = Int(150 * anteil + 30)
            WinkelSetzen(winkel: winkel)
        }

        /**
         Erzeugt den Ableger mit einer Kiste
         */
        func FigurMitKisteFestlegen()
        {
            // Bisherige Darstellung löschen
            EigeneFigurLöschen()
            // Figur mit Kiste darstellen
            FigurteilFestlegenRechteck(x: 0, y: -10, breite: 100, höhe: 20, farbe: "schwarz")
            FigurteilFestlegenEllipse(x: -10, y: -10, breite: 20, höhe: 20, farbe: "schwarz")
            FigurteilFestlegenRechteck(x: 100, y: -25, breite: 50, höhe: 50, farbe: "braun")
        }

        /**
         Erzeugt den Ableger ohe Kiste
         */
        func FigurOhneKisteFestlegen()
        {
            // Bisherige Darstellung löschen
            EigeneFigurLöschen()
            // Figur ohne Kiste darstellen
            FigurteilFestlegenRechteck(x: 0, y: -10, breite: 100, höhe: 20, farbe: "schwarz")
            FigurteilFestlegenEllipse(x: -10, y: -10, breite: 20, höhe: 20, farbe: "schwarz")
        }
    }
}

/**
 * Verbraucher von Kisten.
 *
 * @author Johannes Neumeyer
 * @version 1.0
 */
class Verbraucher: Thread
{
    /** Die Produktionszeit pro Kiste in Millisekunden */
    private var zeit: Int
    /** Der Zwischenspeicher */
    private var speicher: Speicher
    /** Die Darstellung des Verbrauchers **/
    private var verbraucherSymbol: VerbraucherSymbol

    /**
    Konstruktor für Objekte der Klasse Verbraucher
     - parameters:
         - zeitNeu: Produktionszeit für die Kisten
         - speicherNeu: der Speicher, aus dem geholt werden soll
    */
    init(zeitNeu: Int, speicherNeu: Speicher)
    {
        zeit = zeitNeu
        speicher = speicherNeu
        verbraucherSymbol = VerbraucherSymbol()
    }

    /**
    Die Arbeitsmethode des Threads.
     */
    override func main()
    {
        while true
        {
            LeerHinDarstellen()
            var kNeu = speicher.Holen()
            
            while kNeu == nil
            {
                LeerZurückDarstellen()
                LeerHinDarstellen()
                kNeu = speicher.Holen()
            }
            BeladenZurückDarstellen()
            Einlagern(kisteNeu: kNeu!)
        }
    }

    /**
    Lagert Kisten in der gegebenen Zeit.
    - parameters:
        - kisteNeu: die zu lagernde Kiste
     */
    private func Einlagern(kisteNeu: Kiste)
    {
        for zähler in 0 ..< 10
        {
            kisteNeu.SichtbarkeitSetzen(sichtbar: true)
            kisteNeu.PositionSetzen(x: 625 + 5 * (zähler + 1), y: 200)
            kisteNeu.GrößeSetzen(breite: 50 - 5 * (zähler + 1), höhe: 50)

            Thread.sleep(forTimeInterval: Double(zeit) / 10000.0)
        }
    }

    /**
    Stellt den Verbraucher ohne Kiste auf dem Weg zum Abholplatz dar.
     */
    private func LeerHinDarstellen()
    {
        verbraucherSymbol.FigurOhneKisteFestlegen();
        for  i in 0 ..< 10
        {
            verbraucherSymbol.LeerHin(anteil: Double(i) / 9)
            Thread.sleep(forTimeInterval: Double(zeit) / 10000.0)
        }
    }

    /**
     Stellt den Verbraucher ohne Kiste auf dem Weg vom Abholplatz zurück dar.
    */
    private func LeerZurückDarstellen()
    {
        verbraucherSymbol.FigurOhneKisteFestlegen();
        for i in 0 ..< 10
        {
            verbraucherSymbol.LeerZurück(anteil: Double(i) / 9)
            Thread.sleep(forTimeInterval: Double(zeit) / 10000.0)
        }
    }

    /**
    Stellt den Verbraucher mit Kiste auf dem Weg vom Abholplatz zurück dar.
     */
    private func BeladenZurückDarstellen()
    {
        verbraucherSymbol.FigurMitKisteFestlegen();
        for i in 0 ..< 10
        {
            verbraucherSymbol.BeladenZurück(anteil: Double(i) / 9)
            Thread.sleep(forTimeInterval: Double(zeit) / 10000.0)
        }
        verbraucherSymbol.FigurOhneKisteFestlegen();
    }

    /**
    Darstellung des Verbrauchers

    - author: Johannes Neumeyer
    - version: 1.0
    */
    class VerbraucherSymbol: Figur
    {
        /**
         Erzeugt das Symbol in Startstellung
         */
        override init()
        {
            super.init()
            FigurOhneKisteFestlegen()
            GrößeSetzen(größe: 100)
            PositionSetzen(x: 525, y: 225)
            LeerZurück(anteil: 1)
        }

        /**
        Schwenkt den leeren Verbraucherarm Richtung Ablage
        - parameters:
            - anteil: Winkelanzeil des Schwenkbereichs
        */
        func LeerHin(anteil: Double)
        {
            // Anteil an 180° in Winkel umrechnen und Winkel setzen
            let winkel = Int(160 * anteil)
            WinkelSetzen(winkel: winkel)
        }

        /**
        Schwenkt den leeren Verbraucherarm Richtung Einlagerung
         - parameters:
             - anteil: Winkelanzeil des Schwenkbereichs
        */
        func LeerZurück(anteil: Double)
        {
            // Anteil an 180° in Winkel umrechnen und Winkel setzen
            let winkel = Int(160 - 160 * anteil)
            WinkelSetzen(winkel: winkel)
        }

        /**
        Schwenkt den beladenen Verbraucherarm Richtung Einlagerung
         - parameters:
             - anteil: Winkelanzeil des Schwenkbereichs
        */
        func BeladenZurück(anteil: Double)
        {
            // Anteil an 180° in Winkel umrechnen und Winkel setzen
            let winkel = Int(160 - 160 * anteil)
            WinkelSetzen(winkel: winkel)
        }

        /**
        Erzeugt den Konsumenten mit einer Kiste
        */
        func FigurMitKisteFestlegen()
        {
            // Bisherige Darstellung löschen
            EigeneFigurLöschen()
            // Figur mit Kiste darstellen
            FigurteilFestlegenRechteck(x: 0, y: -10, breite: 100, höhe: 20, farbe: "schwarz")
            FigurteilFestlegenEllipse(x: -10, y: -10, breite: 20, höhe: 20, farbe: "schwarz")
            FigurteilFestlegenRechteck(x: 100, y: -25, breite: 50, höhe: 50, farbe: "braun")
        }

        /**
         * Erzeugt den Konsumenten ohne Kiste
         */
        func FigurOhneKisteFestlegen()
        {
            // Bisherige Darstellung löschen
            EigeneFigurLöschen()
            // Figur ohne Kiste darstellen
            FigurteilFestlegenRechteck(x: 0, y: -10, breite: 100, höhe: 20, farbe: "schwarz")
            FigurteilFestlegenEllipse(x: -10, y: -10, breite: 20, höhe: 20, farbe: "schwarz")
        }
    }
}


/**
Erzeuger-Verbraucher-Problem

- author: Johannes Neumeyer
- version: 1.0
 */
class ErzeugerVerbraucherProblem
{

    /**
    Konstruktor für Objekte der Klasse ErzeugerVerbraucherProblem
     */
    init()
    {
        let speicher = Speicher()
        let erzeuger = Erzeuger(zeitNeu: 500, speicherNeu: speicher)
        let verbraucher = Verbraucher(zeitNeu: 2500, speicherNeu: speicher)
        erzeuger.start()
        verbraucher.start()
    }
}

ErzeugerVerbraucherProblem()
