/**
Grafikprojekt - Einfache graphische Elemente auf einer View
- Author: Albert Wiedemann
- Version: 2.0
*/

import Foundation
#if os(macOS)
import Cocoa
#elseif os(iOS)
import UIKit
#endif

#if os(macOS)
typealias SystemFarbe = NSColor
typealias View = NSView
#elseif os(iOS)
typealias SystemFarbe = UIColor
typealias View = UIView
#endif
import PlaygroundSupport

public let anzeigebreite = 800.0
public let anzeigeHöhe = 600.0
#if os(macOS)
let zusatzHöhe = 30.0
#elseif os(iOS)
let zusatzHöhe = 60.0
#endif

/**
Basisklasse für die Zeichenobjekte
*/
class ZeichenObjekt
{
    /** Farbe der Figur */
    var color: SystemFarbe
    /** x-Koordinate der linken, oberen Ecke der Figur */
    var x: CGFloat
    /** y-Koordinate der linken, oberen Ecke der Figur */
    var y: CGFloat
    /** Breite der Figur */
    var breite: CGFloat
    /** Höhe der Figur */
    var höhe: CGFloat
    /** Sichtbarkeit der Figur */
    var sichtbar: Bool
    /** Drehwinkel (mathematisch positiver Drehsinn) des Symbols. */
    var winkel: Int
    /** Die Form des Grafiksymbols. */
    var form: CGPath
    /** Farbe Hellgelb. */
    let hellgelb = SystemFarbe(red: 1, green: 1, blue: 0.5, alpha: 1)
    /** Farbe Hellgrün. */
    let hellgrün = SystemFarbe(red: 0.5, green: 1, blue: 0.5, alpha: 1)
    /** Testcontext für Schneiden. */
    let testContext: CGContext


    /**
    Legt ein rotes Objekt der Breite 100 und der Höhe 100  in der linken oberen Ecke an.
    Bei Bedarf wird die View initialisiert
    */
    init ()
    {
        x = 10.0
        y = 10.0
        breite = 100.0
        höhe = 100.0
        winkel = 0
        color = SystemFarbe.red
        sichtbar = true
        form = CGMutablePath()
        testContext = CGContext(data: nil, width: Int(anzeigebreite), height: Int(anzeigeHöhe), bitsPerComponent: 8, bytesPerRow: 0, space: CGColorSpace(name: CGColorSpace.sRGB)!, bitmapInfo: CGImageAlphaInfo.premultipliedLast.rawValue)!
        FormErzeugen()
        if MeineView.v == nil
        {
            PlaygroundPage.current.liveView = GesamtFenster(frame: CGRect (x: 0.0, y: 0.0, width: anzeigebreite, height: anzeigeHöhe + zusatzHöhe))
        }
        MeineView.v.figurenLock.lock()
        MeineView.v.figuren.append(self)
        MeineView.v.figurenLock.unlock()
        MeineView.v.NeuZeichnen()
    }
    
    /**
    Normiert den Winkel auf Werte im Bereich [0; 360[
    - parameters:
    	- winkel der Eingabewinkel
    - returns: der normierte Winkel
    */
    func WinkelNormieren(winkel: Int) -> Int
    {
        var w = winkel
        while w < 0
        {
            w += 360
        }
        return w % 360
    }
    
    /**
    Erstellt die Form des Objekts.
    Muss in den Unterklassen überschrieben werden.
    */
    func FormErzeugen()
    {
        fatalError ("FormErzeugen wurde nicht überschrieben")
    }
    
    /**
    Zeichnet die Figur in den gegebenen Context.
    - parameters:
        - context: Grafikkontext
    */
    func Zeichnen (context: CGContext)
    {
        context.setFillColor(color.cgColor)
        context.addPath(form)
        context.fillPath()
    }

    /**
    Bestimmt die RGB-Farbe für den gegeben String, erkannt werden die Farben
    rot, grün, hellgrün, blau, gelb, cyan, magente, weiß, schwarz, grau, braun und orange
    - parameters:
        - farbe: Die Farbe der Figur
    - returns: RGB-Farbe
    */
    func FarbeCodieren (farbe: String) -> SystemFarbe
    {
        switch (farbe.lowercased())
        {
            case "rot":
                return SystemFarbe.red
            case "gelb":
                return SystemFarbe.yellow
            case "grün", "gruen":
                return SystemFarbe.green
            case "hellgrün", "hellgruen":
                return hellgrün
            case "hellgelb":
                return hellgelb
            case "blau":
                return SystemFarbe.blue
            case "cyan":
                return SystemFarbe.cyan
            case "magenta":
                return SystemFarbe.magenta
            case "weiß", "weiss":
                return SystemFarbe.white
            case "schwarz":
                return SystemFarbe.black
            case "grau":
                return SystemFarbe.gray
            case "braun":
                return SystemFarbe.brown
            case "orange":
                return SystemFarbe.orange
            default:
                return SystemFarbe.black
        }
    }

    /**
    Setzt die Farbe des Objekts, erkannt werden die Farben
    rot, grün, hellgrün, blau, gelb, cyan, magente, weiß, schwarz, grau, braun und orange
    - parameters:
        - farbe: Die Farbe der Figur
    */
    func FarbeSetzen (farbe: String)
    {
        FarbeSetzen(farbe: FarbeCodieren(farbe: farbe))
    }

    /**
    Setzt die Farbe des Objekts
    - parameters:
        - farbe: Die (neue) Farbe der Figur
    */
    func FarbeSetzen (farbe: SystemFarbe)
    {
        color = farbe
        MeineView.v.NeuZeichnen()
    }

    /**
    Setzt die Koordinaten der Figur auf die angegebenen Koordinaten
    - parameters:
        - x: x-Koordinate der linken oberen Ecke bzw. der Spitze bzw. des Mittelpunkts
        - y: y-Koordinate der linken oberen Ecke bzw. der Spitze bzw. des Mittelpunkts
    */
    func PositionSetzen (x: Int, y: Int)
    {
        self.x = CGFloat (x)
        self.y = CGFloat (y)
        FormErzeugen()
        MeineView.v.NeuZeichnen()
    }
    
    /**
    Setzt die Größe der Figur auf die angegebene Breite und Höhe
    - parameters:
        - breite: Breite des (umgebenden) Rechtecks
        - höhe: Höhe des (umgebenden) Rechtecks
    */
    func GrößeSetzen (breite: Int, höhe: Int)
    {
        self.breite = CGFloat (breite)
        self.höhe = CGFloat (höhe)
        FormErzeugen()
        MeineView.v.NeuZeichnen()
    }

    /**
    Setzt die Sichtbarkeit der Figur
    - parameters:
        - sichtbar: die Figur ist sichtbar, wenn wahr
    */
    func SichtbarkeitSetzen (sichtbar: Bool)
    {
        self.sichtbar = sichtbar
        MeineView.v.NeuZeichnen()
    }
    
    /**
    Setzt den Drehwinkel des Objekts
    - parameters:
        - breite: Breite des (umgebenden) Rechtecks
        - höhe: Höhe des (umgebenden) Rechtecks
    */
    func WinkelSetzen (winkel: Int)
    {
        self.winkel = WinkelNormieren(winkel: winkel)
        FormErzeugen()
        MeineView.v.NeuZeichnen()
    }
    
    /**
    Berechnet den Drehwinkel gemäß den Konventionen des Graphik-Frameworks.
    Für iOS: Winkel in Radians, positive Drehrichtung im Uhrzeiger.
    Für macOS: Winkel in Radians, positive Drehrichtung im Uhrzeiger wegen der Spiegelung des Koordinatensystems.
    - parameters:
        - winkel: der mathematische Drehwinkel
    */
    func DrehwinkelGeben (winkel: Int) -> CGFloat
    {
        return -CGFloat.pi * CGFloat(winkel) / 180.0;
    }
    
    /**
    Testet, ob der angegebene Punkt in der Figur ist.
    - parameters:
        - x: x-Koordinate des Punktes
        - y: y-Koordinate des Punktes
    - returns: wahr, wenn der Punkt in der Figur liegt.
    */
    func IstInnerhalb(x: CGFloat, y: CGFloat) -> Bool
    {
        return form.contains(CGPoint(x: x, y: y))
    }
    
    /**
    Testet, ob der angegebene Punkt in der Figur ist.
    - parameters:
        - x: x-Koordinate des Punktes
        - y: y-Koordinate des Punktes
    - returns: wahr, wenn der Punkt in der Figur liegt.
    */
    func Schneidet (wen: CGPath) -> Bool
    {
        testContext.saveGState()
        testContext.clip(to: CGRect(x: 0.0, y: 0.0, width: 1000, height: 800))
        testContext.beginPath()
        testContext.addPath(wen)
        testContext.clip()
        testContext.beginPath()
        testContext.addPath(form)
        testContext.clip()
        let res = (testContext.boundingBoxOfClipPath.width != 0) || (testContext.boundingBoxOfClipPath.height != 0)
        testContext.restoreGState()
        return res
    }

    /**
    Entfernt das Objekt aus dem Zeichenfenster.
    */
    func Entfernen ()
    {
        MeineView.v.figurenLock.lock()
        MeineView.v.figuren.remove(at: MeineView.v.figuren.firstIndex(where: {$0 === self})!)
        MeineView.v.figurenLock.unlock()
    	MeineView.v.NeuZeichnen()
    }

    /**
    Bringt das Objekt eine Ebene nach vorn.
    */
    func NachVornBringen()
    {
        MeineView.v.figurenLock.lock()
        let index = MeineView.v.figuren.firstIndex(where: {$0 === self})!
        if index < MeineView.v.figuren.count - 1
        {
            MeineView.v.figuren[index] = MeineView.v.figuren[index+1]
            MeineView.v.figuren[index+1] = self
            MeineView.v.NeuZeichnen()
        }
        MeineView.v.figurenLock.unlock()
    }
    
    /**
    Bringt das Objekt in die vorderste Ebene.
    */
    func GanzNachVornBringen()
    {
        MeineView.v.figurenLock.lock()
        let index = MeineView.v.figuren.firstIndex(where: {$0 === self})!
        if index < MeineView.v.figuren.count - 1
        {
            MeineView.v.figuren.remove(at: index)
            MeineView.v.figuren.append(self)
            MeineView.v.NeuZeichnen()
        }
        MeineView.v.figurenLock.unlock()
    }
    
    /**
    Bringt das Objekt eine Ebene nach hinten.
    */
    func NachHintenBringen()
    {
        MeineView.v.figurenLock.lock()
        let index = MeineView.v.figuren.firstIndex(where: {$0 === self})!
        if index > 0
        {
            MeineView.v.figuren[index] = MeineView.v.figuren[index-1]
            MeineView.v.figuren[index-1] = self
            MeineView.v.NeuZeichnen()
        }
        MeineView.v.figurenLock.unlock()
    }
    
    /**
    Bringt das Objekt in die hinterste Ebene.
    */
    func GanzNachHintenBringen()
    {
        MeineView.v.figurenLock.lock()
        let index = MeineView.v.figuren.firstIndex(where: {$0 === self})!
        if index > 0
        {
            MeineView.v.figuren.remove(at: index)
            MeineView.v.figuren.insert(self, at: 0)
            MeineView.v.NeuZeichnen()
        }
        MeineView.v.figurenLock.unlock()
    }
    
    /**
	Liefert das Darstellungssymbol für ein Zeichenobjekt
	- parameters:
		- objekt: Das gegebene Objekt
    - returns: Referenz auf das interne Objekt oder nil, wenn das Objekt kein Zeichenobjekt repräsentiert.
    */
    func SymbolGeben(objekt: AnyObject) -> ZeichenObjekt?
    {
        switch objekt
        {
        case let s as Rechteck:
            return s.symbol
        case let s as Dreieck:
            return s.symbol
        case let s as Kreis:
            return s.symbol
        case let s as Text:
            return s.symbol
        case let s as Turtle:
            return s.symbol
        case let s as Figur:
            return s.symbol
        default:
            return nil
        }
    }
}

/**
Stellt ein Rechteck dar.
Die Position bezieht sich auf die linke, obere Ecke
*/
class RechteckIntern: ZeichenObjekt
{
    /**
    Erstellt die Form des Rechtecks.
    */
    override func FormErzeugen()
    {
        var rot = CGAffineTransform(translationX: x + breite / 2.0, y: y + höhe / 2.0).rotated(by: DrehwinkelGeben(winkel: winkel))
        form = CGPath(rect: CGRect(x: -breite / 2, y: -höhe / 2, width: breite, height: höhe), transform: &rot)
    }
}

/**
Stellt ein Dreieck dar.
Die Position bezieht sich auf die Spitze des Dreiecks
*/
class DreieckIntern: ZeichenObjekt
{
    /**
    Erstellt die Form des Dreiecks.
    */
    override func FormErzeugen()
    {
        let rot = CGAffineTransform(translationX: x + breite / 2.0, y: y + höhe / 2.0).rotated(by: DrehwinkelGeben(winkel: winkel))
        let f = CGMutablePath()
        f.move(to: CGPoint (x: -breite / 2.0, y: höhe / 2.0), transform: rot)
        f.addLine(to: CGPoint (x: breite / 2.0, y: höhe / 2.0), transform: rot)
        f.addLine(to: CGPoint (x: 0.0, y: -höhe / 2.0), transform: rot)
        f.addLine(to: CGPoint (x: -breite / 2.0, y: höhe / 2.0), transform: rot)
        form = f
    }
}

/**
Stellt eine Ellipse dar.
Die Position bezieht sich auf die linke obere Ecke des umgebenden Rechtecks.
*/
class EllipseIntern: ZeichenObjekt
{
    /**
    Erstellt die Form der Ellipse.
    */
    override func FormErzeugen()
    {
        var rot = CGAffineTransform(translationX: x + breite / 2.0, y: y + höhe / 2.0).rotated(by: DrehwinkelGeben(winkel: winkel))
        form = CGPath(ellipseIn: CGRect(x: -breite / 2, y: -höhe / 2, width: breite, height: höhe), transform: &rot)
    }
}

/**
Verwaltet einen Text
*/
class TextIntern: ZeichenObjekt
{
    /** Der aktuelle Text. */
    var text: String = ""
    /** Die aktuelle Textgröße. */
    var size: CGFloat
    
    /**
    Belegt Text und Größe mit Standardwertern
    */
    override init()
    {
        text = "Text"
        size = 12
        super.init()
        color = SystemFarbe.black
    }
    
    /**
    Erstellt die Form des Textes.
    Dummy, legt ein unendlich kleines Rechteck am Ende der Welt an.
    */
    override func FormErzeugen()
    {
        form = CGPath(rect: CGRect(x: Int.min, y: Int.min, width: 0, height: 0), transform: nil)
    }
    
    /**
    Testet, ob der angegebene Punkt in der Figur ist.
    - parameters:
        - x: x-Koordinate des Punktes
        - y: y-Koordinate des Punktes
    - returns: falsch
    */
    override func IstInnerhalb(x: CGFloat, y: CGFloat) -> Bool
    {
        return false
    }
    
    /**
    Setzt den aktuellen Text.
    - parameters:
        - t: der neue Text
    */
    func TextSetzen (t: String)
    {
        text = t
        MeineView.v.NeuZeichnen()
    }
    
    /**
    Setzt die Größe des Textes.
    - parameters:
        größe: die neue Textgröß
    */
    func TextGrößeSetzen (größe: Int)
    {
        size = CGFloat(größe)
        MeineView.v.NeuZeichnen()
    }
    
    /**
    Vergrößert den Text.
    */
    func TextVergrößern()
    {
        if size <= 10
        {
            size += 1
        }
        else if size <= 40
        {
            size += 2
        }
        else
        {
            size += 4
        }
        MeineView.v.NeuZeichnen()
    }
    
    /**
    Verkleinert den Text.
    */
    func TextVerkleinern()
    {
        if size <= 10
        {
            size -= 1
        }
        else if size <= 40
        {
            size -= 2
        }
        else
        {
            size -= 4
        }
        MeineView.v.NeuZeichnen()
    }

    /**
    Zeichnet die Figur in den gegebenen Context.
    - parameters:
        - context: Grafikkontext
    */
    override func Zeichnen (context: CGContext)
    {
        context.setStrokeColor(color.cgColor)
        let font = CTFontCreateWithName (("Helvetica" as CFString?)!, size, nil)
        let attrString = NSAttributedString(string: text, attributes: [(NSAttributedString.Key(rawValue: kCTForegroundColorAttributeName as String as String)):color, (NSAttributedString.Key(rawValue: kCTFontAttributeName as String as String)):font])
        let line = CTLineCreateWithAttributedString(attrString)
        let lineRect = CTLineGetImageBounds(line, context)
        context.textPosition = CGPoint(x: x, y: y)
        context.saveGState()
        context.translateBy(x: x + lineRect.width/2, y: y - lineRect.height/2)
        context.rotate(by: DrehwinkelGeben(winkel: winkel))
        context.translateBy(x: -(x + lineRect.width/2), y: -(y - lineRect.height/2))
        CTLineDraw(line, context)
        context.restoreGState()
    }
}

/**
Oberklasse für alle Elemente einer Figur (Figur, Turtle).
*/
class FigurenElement
{
    let xe: CGFloat
    let ye: CGFloat
    let breite: CGFloat
    let höhe: CGFloat
    let c:SystemFarbe

	/**
	Legt die Attribute fest.
	- parameters:
		- xe: die x-Koordiante der linken obren Ecke der Figur
    	- ye: die y-Koordiante der linken obren Ecke der Figur
    	- breite: die Breite der Figur
    	- höhe: die Höhe der Figur
    	- c: die Farbe der Figur
	*/
    init(xe: CGFloat, ye: CGFloat, breite: CGFloat, höhe: CGFloat, c: SystemFarbe)
    {
        self.xe = xe
        self.ye = ye
        self.breite = breite
        self.höhe = höhe
        self.c = c
    }

	/**
	Zeichnet das Figurenelement.
	- parameters:
		- context: der Grafikkontext
		- größe: die tatsächliche Größe der Figur
	*/
    func ElementZeichnen (context: CGContext, größe: CGFloat)
    {
        fatalError("ElementZeichnen muss überschrieben werden")
    }
    
    /**
    Fügt den Pfadteil dieser Komponente zum umgebenden Pfad hinzu.
    - parameters:
        - p: der Gesamtpfad
        - größe: die tatsächliche Größe der Figur
        - trans: die aktuelle affine Transformation
    */
    func ElementZuForm (p: CGMutablePath, größe: CGFloat, trans: CGAffineTransform)
    {
        fatalError("ElementZuForm muss überschrieben werden")
    }
}

/**
Ein rechteckiges Figurenelement.
*/
class FigurenElementRechteck: FigurenElement
{
    /**
    Zeichnet das Figurenelement.
    - parameters:
        - context: der Grafikkontext
        - größe: die tatsächliche Größe der Figur
    */
    override func ElementZeichnen (context: CGContext, größe: CGFloat)
    {
        context.setFillColor(c.cgColor)
        context.fill(CGRect(x: größe*xe/100.0, y: größe*ye/100.0, width: größe*breite/100.0, height: größe*höhe/100.0))
    }
    
    /**
    Fügt den Pfadteil dieser Komponente zum umgebenden Pfad hinzu.
    - parameters:
        - p: der Gesamtpfad
        - größe: die tatsächliche Größe der Figur
        - trans: die aktuelle affine Transformation
    */
    override func ElementZuForm (p: CGMutablePath, größe: CGFloat, trans: CGAffineTransform)
    {
        p.addRect(CGRect(x: größe*xe/100.0, y: größe*ye/100.0, width: größe*breite/100.0, height: größe*höhe/100.0), transform: trans)
    }
}

/**
Ein elliptisches Figurenelement.
*/
class FigurenElementEllipse: FigurenElement
{
    /**
    Zeichnet das Figurenelement.
    - parameters:
        - context: der Grafikkontext
        - größe: die tatsächliche Größe der Figur
    */
    override func ElementZeichnen (context: CGContext, größe: CGFloat)
    {
        context.setFillColor(c.cgColor)
        context.fillEllipse(in: CGRect(x: größe*xe/100.0, y: größe*ye/100.0, width: größe*breite/100.0, height: größe*höhe/100.0))
    }
    
    /**
    Fügt den Pfadteil dieser Komponente zum umgebenden Pfad hinzu.
    - parameters:
        - p: der Gesamtpfad
        - größe: die tatsächliche Größe der Figur
        - trans: die aktuelle affine Transformation
    */
    override func ElementZuForm (p: CGMutablePath, größe: CGFloat, trans: CGAffineTransform)
    {
        p.addEllipse(in: CGRect(x: größe*xe/100.0, y: größe*ye/100.0, width: größe*breite/100.0, height: größe*höhe/100.0), transform: trans)
    }
}

/*
Ein Figurenelement begrenzt durch das angegebene Polygon.
*/
class FigurenElementPolygon: FigurenElement
{
	let path: CGMutablePath

    /**
    Legt die Attribute fest.
    - parameters:
        - xe: die x-Koordiante der linken obren Ecke der Figur
        - ye: die y-Koordiante der linken obren Ecke der Figur
        - breite: die Breite der Figur
        - höhe: die Höhe der Figur
        - c: die Farbe der Figur
    */
    init(p: [CGPoint], c: SystemFarbe)
    {
        path = CGMutablePath()
        path.addLines(between: p)
        super.init(xe: path.boundingBoxOfPath.origin.x, ye: path.boundingBoxOfPath.origin.y, breite: path.boundingBoxOfPath.size.width, höhe: path.boundingBoxOfPath.size.height, c: c)
        path.closeSubpath()
    }

	
    /**
    Zeichnet das Figurenelement.
    - parameters:
        - context: der Grafikkontext
        - größe: die tatsächliche Größe der Figur
    */
    override func ElementZeichnen (context: CGContext, größe: CGFloat)
    {
        context.setFillColor(c.cgColor)
        var trans = CGAffineTransform(scaleX: größe / 100, y: größe / 100)
        context.beginPath()
        context.addPath(path.copy(using: &trans)!)
        context.fillPath()
    }
    
    /**
    Fügt den Pfadteil dieser Komponente zum umgebenden Pfad hinzu.
    - parameters:
        - p: der Gesamtpfad
        - größe: die tatsächliche Größe der Figur
        - trans: die aktuelle affine Transformation
    */
    override func ElementZuForm (p: CGMutablePath, größe: CGFloat, trans: CGAffineTransform)
    {
        p.addPath(path, transform: CGAffineTransform(scaleX: größe / 100, y: größe / 100).concatenating(trans))
    }
}

/**
Verwaltet eine Turtle
Das Objekt dieser Klasse zeichnet auch den Weg der Turtle
*/
class TurtleIntern: ZeichenObjekt
{
    private class LinienElement
    {
        /** x-Koordinate des Startpunktes. */
        private let xStart: CGFloat
        /** y-Koordinate des Startpunktes. */
        private let yStart: CGFloat
        /** x-Koordinate des Endpunktes. */
        private let xEnde: CGFloat
        /** y-Koordinate des Endpunktes. */
        private let yEnde: CGFloat
        /** Farbe des LinienElements. */
        private let c: SystemFarbe
        
        /**
        Belegt die Attributwerte
        - parameters:
            - xStart X-Koordinate des Linienstarts
            - yStart Y-Koordinate des Linienstarts
            - xEnde X-Koordinate des Linienendes
            - yEnde Y-Koordinate des Linienendes
            - c: Farbe der Linie
        */
        init(xStart: CGFloat, yStart: CGFloat, xEnde: CGFloat, yEnde: CGFloat, c: SystemFarbe)
        {
            self.xStart = xStart
            self.yStart = yStart
            self.xEnde = xEnde
            self.yEnde = yEnde
            self.c = c
        }
        
        func Zeichnen (context: CGContext)
        {
            context.setStrokeColor(c.cgColor)
            context.move(to: CGPoint(x: xStart, y: yStart))
            context.addLine(to: CGPoint(x: xEnde, y: yEnde))
            context.strokePath()
        }
    }

    /** Startkoordinate der Turtle. */
    private var homeX: CGFloat
    /** Startkoordinate der Turtle. */
    private var homeY: CGFloat
    /** Startwinkel der Turtle. */
    private var homeWinkel: Int
    /** Stiftposition. */
    var stiftUnten: Bool
    /** Die Sichtbarkeit des Turtle-Symbols. */
    var symbolSichtbar: Bool
    /** Linienelemente. */
    private var linien: [LinienElement]
    /** Standardfigur für Turtle. */
    var standardFigur: [FigurenElement]

    /**
    Legt die Turtle mit Startpunkt (100|200) in Richtung 0˚ an.
    */
    override init()
    {
        homeX = 100
        homeY = 200
        homeWinkel = 0
        stiftUnten = true
        symbolSichtbar = true
        linien = [LinienElement]()
        standardFigur = [FigurenElement]()
        super.init()
        x = 100
        y = 200
        höhe = 40
        breite = 40
        homeWinkel=winkel
        color = SystemFarbe.black
        StandardfigurErzeugen()
        FormErzeugen()
    }
    
    /**
    Baut die Standardfigur aus den Elementen auf.
    */
    private func StandardfigurErzeugen()
    {
        //Kopf
        standardFigur.append(FigurenElementEllipse(xe: 50, ye: -12.5, breite: 25, höhe: 25, c: SystemFarbe.green))
        //Beine
        standardFigur.append(FigurenElementEllipse(xe: 22.5, ye: -32.5, breite: 12.5, höhe: 17.5, c: SystemFarbe.green))
        standardFigur.append(FigurenElementEllipse(xe: 40.0, ye: -32.5, breite: 12.5, höhe: 17.5, c: SystemFarbe.green))
        standardFigur.append(FigurenElementEllipse(xe: 22.5, ye: 15.0, breite: 12.5, höhe: 17.5, c: SystemFarbe.green))
        standardFigur.append(FigurenElementEllipse(xe: 40.0, ye: 15.0, breite: 12.5, höhe: 17.5, c: SystemFarbe.green))
        //Augen
        standardFigur.append(FigurenElementRechteck(xe: 67.5, ye: -10.0, breite: 5.0, höhe: 7.5, c: color))
        standardFigur.append(FigurenElementRechteck(xe: 67.5, ye: 2.5, breite: 5.0, höhe: 7.5, c: color))
        //Schwanz
        standardFigur.append(FigurenElementEllipse(xe: 0, ye: -3.75, breite: 25, höhe: 7.5, c: color))
        //Rumpf
        standardFigur.append(FigurenElementEllipse(xe: 7.5, ye: -23.75, breite: 57.5, höhe: 47.5, c: SystemFarbe.brown));
    }

    /**
    Erstellt die Form der Turtle.
    */
    override func FormErzeugen()
    {
        let rot = CGAffineTransform(translationX: x, y: y).rotated(by: DrehwinkelGeben(winkel: winkel))
        let f = CGMutablePath()
        let größe = breite > höhe ? höhe : breite
        if standardFigur.count > 0
        {
        	for element in standardFigur
        	{
        		element.ElementZuForm(p: f, größe: größe, trans: rot)
            }
        }
        form = f
    }

    /**
    Setzt die Turtle wieder an ihre Ausgangsposition.
    */
    func ZumStartpunktGehen()
    {
        x = homeX
        y = homeY
        winkel = homeWinkel
        MeineView.v.NeuZeichnen()
    }
    
    /**
    Bewegt die Turtle nach vorne.
    - parameters:
        - länge: anzahl der Längeneinheiten
    */
    func Gehen (länge: CGFloat)
    {
        let xNeu = x + cos(DrehwinkelGeben (winkel: winkel)) * länge
        let yNeu = y + sin(DrehwinkelGeben (winkel: winkel)) * länge
        if stiftUnten
        {
            linien.append(LinienElement(xStart: x, yStart: y, xEnde: xNeu, yEnde: yNeu, c: color))
        }
        x = xNeu
        y = yNeu
        FormErzeugen()
        MeineView.v.NeuZeichnen()
    }
    
    /**
    Dreht die Turtle
    - parameters:
        - grad: Drehwinkel im Gradmass
    */
    func Drehen (grad: Int)
    {
        winkel = WinkelNormieren(winkel: winkel + grad)
        FormErzeugen()
        MeineView.v.NeuZeichnen()
    }
    
    /**
    Versetzt Zeichenfläche und Turtle in den Ausgangszustand
    */
    func Löschen()
    {
        linien.removeAll()
        ZumStartpunktGehen()
    }
    
    /**
    Turtle wechselt in den Modus "nicht zeichnen"
    */
    func StiftHeben()
    {
        stiftUnten = false
    }
    
    /**
    Turtle wechselt in den Modus "zeichnen"
    */
    func StiftSenken()
    {
        stiftUnten = true
    }

    /**
    Schaltet die Sichtbarkeit des Turtlesymbols ein oder aus.
    Erlaubte Parameterwerte: true, false
    - parameters:
        - sichtbar: (neue) Sichtbarkeit des Turtlesymbols
     */
    func SichtbarkeitFürSymbolSetzen (sichtbar: Bool)
    {
        symbolSichtbar = sichtbar
        MeineView.v.NeuZeichnen()
    }

    /**
    Testet, ob die Turtle eine (sichtbare) Figur berührt.
    - returns: true, wenn die Turtlekoordinaten innerhalb einer Grafikfigur sind
     */
    func Berührt() -> Bool
    {
        MeineView.v.figurenLock.lock()
        let res = MeineView.v.figuren.first(where: {($0 !== self) && $0.IstInnerhalb(x: x, y: y) && $0.sichtbar && (!($0 is TurtleIntern) || ($0 as! TurtleIntern).symbolSichtbar)}) != nil
        MeineView.v.figurenLock.unlock()
        return res
    }

    /**
    Testet, ob die Turtle eine (sichtbare) Figur in der angegebenen Farbe berührt.
    - parameters:
        - farbe: die Farbe, die die berührte Figur haben muss
    - returns: true, wenn die Turtlekoordinaten innerhalb einer Grafikfigur in der angegebenen Farbe sind
     */
    func Berührt(farbe: String) -> Bool
    {
        let color2 = FarbeCodieren(farbe: farbe)
        var ok = false
        MeineView.v.figurenLock.lock()
        for f in MeineView.v.figuren
        {
        	if (f !== self) && f.sichtbar && f.IstInnerhalb(x: x, y: y)
        	{
        		if let t = f as? TurtleIntern
            	{
            		if t.symbolSichtbar
                	{
                    	for e in standardFigur
                    	{
                            let rot = CGAffineTransform(translationX: t.x, y: t.y).rotated(by: DrehwinkelGeben(winkel: t.winkel))
                            let f = CGMutablePath()
                            let größe = t.breite > t.höhe ? t.höhe : t.breite
							e.ElementZuForm(p: f, größe: größe, trans: rot)
                        	if f.contains((CGPoint(x: x, y: y)))
                        	{
                        		ok = color2 == e.c
                            }
                        }
                    }
                }
                else if let t = f as? FigurIntern
                {
                    let figur = t.eigeneFigur.count > 0 ? t.eigeneFigur : t.standardFigur
                    for e in figur
                    {
                        let rot = CGAffineTransform(translationX: t.x, y: t.y).rotated(by: DrehwinkelGeben(winkel: t.winkel))
                        let f = CGMutablePath()
                        let größe = t.breite > t.höhe ? t.höhe : t.breite
                        e.ElementZuForm(p: f, größe: größe, trans: rot)
                        if f.contains((CGPoint(x: x, y: y)))
                        {
                            ok = color2 == e.c
                        }
                    }
                }
               else
                {
                	ok = ok || (color2 == f.color)
                }
            }
        }
        MeineView.v.figurenLock.unlock()
        return ok
    }

    /**
    Testet, ob die Turtle die (sichtbare, ) angegebene Figur berührt.
    - parameters:
        - farbe: die Farbe, die die berührte Figur haben muss
    - returns: true, wenn die Turtlekoordinaten innerhalb einer Grafikfigur in der angegebenen Farbe sind
     */
    func Berührt(object: AnyObject) -> Bool
    {
        if let o = SymbolGeben(objekt: object)
        {
            return (o !== self) && o.IstInnerhalb(x: x, y: y) && o.sichtbar && (!(o is TurtleIntern) || (o as! TurtleIntern).symbolSichtbar)
        }
        else
        {
            return false
        }
    }
    /**
    Zeichnet die Turtle mit ihrer Spur in den gegebenen Context.
    - parameters:
        - context: Grafikkontext
    */
    override func Zeichnen(context: CGContext)
    {
        for l in linien
        {
            l.Zeichnen(context: context)
        }
        if symbolSichtbar
        {
            // Outline
            context.saveGState()
            context.setStrokeColor(SystemFarbe.black.cgColor)
            context.setLineWidth(3.0)
            context.beginPath()
            context.addPath(form)
            context.strokePath()
            // Füllung
            let größe = breite > höhe ? höhe : breite
            context.translateBy(x: x, y: y)
            context.rotate(by: DrehwinkelGeben (winkel: winkel))
            for e in standardFigur
            {
            	e.ElementZeichnen(context: context, größe: größe)
            }
            context.restoreGState()
        }
    }

}

/**
Das Objekt dieser Klasse ist ein in der Gestalt definierbarer Akteur.
*/
class FigurIntern: ZeichenObjekt
{

    /** Startkoordinate der Figur. */
    private var homeX: CGFloat
    /** Startkoordinate der Figur. */
    private var homeY: CGFloat
    /** Startwinkel der Figur. */
    private var homeWinkel: Int
    /** Eigene Figur für Figur. */
    var eigeneFigur: [FigurenElement]
    /** Standardfigur für Figur. */
    var standardFigur: [FigurenElement]

    /**
    Legt die Figur mit Startpunkt (100|200) in Richtung 0˚ an.
    */
    override init()
    {
        homeX = 100
        homeY = 200
        homeWinkel = 0
        eigeneFigur = [FigurenElement]()
        standardFigur = [FigurenElement]()
        super.init()
        x = 100
        y = 200
        höhe = 40
        breite = 40
        homeWinkel=winkel
        color = SystemFarbe.black
        StandardfigurErzeugen()
        FormErzeugen()
    }
    
    /**
    Baut die Standardfigur aus den Elementen auf.
    */
    private func StandardfigurErzeugen()
    {
            standardFigur.append(FigurenElementPolygon (p: [CGPoint(x: -50, y: -50), CGPoint(x: 50, y: 0), CGPoint(x: -50, y: 50)], c: SystemFarbe.yellow))
            standardFigur.append(FigurenElementEllipse(xe: -10, ye: -10, breite: 20, höhe: 20, c: SystemFarbe.blue))
    }

    /**
    Erstellt die Form der Figur.
    */
    override func FormErzeugen()
    {
        let rot = CGAffineTransform(translationX: x, y: y).rotated(by: DrehwinkelGeben(winkel: winkel))
        let f = CGMutablePath()
        let größe = breite > höhe ? höhe : breite
        if standardFigur.count > 0
        {
            let figur = eigeneFigur.count > 0 ? eigeneFigur : standardFigur
            for element in figur
            {
                element.ElementZuForm(p: f, größe: größe, trans: rot)
            }
        }
        form = f
    }

    /**
    Setzt die Figur wieder an ihre Ausgangsposition.
    */
    func ZumStartpunktGehen()
    {
        x = homeX
        y = homeY
        winkel = homeWinkel
        MeineView.v.NeuZeichnen()
    }
    
    /**
    Bewegt die Figur nach vorne.
    - parameters:
        - länge: anzahl der Längeneinheiten
    */
    func Gehen (länge: CGFloat)
    {
        x = x + cos(DrehwinkelGeben (winkel: winkel)) * länge
        y = y + sin(DrehwinkelGeben (winkel: winkel)) * länge
        FormErzeugen()
        MeineView.v.NeuZeichnen()
    }
    
    /**
    Dreht die Figur
    - parameters:
        - grad: Drehwinkel im Gradmass
    */
    func Drehen (grad: Int)
    {
        winkel = WinkelNormieren(winkel: winkel + grad)
        FormErzeugen()
        MeineView.v.NeuZeichnen()
    }

    /**
    Testet, ob die Figur eine (sichtbare) Grafik-Figur berührt.
    - returns: true, wenn die Figurkoordinaten innerhalb einer Grafikfigur sind
     */
    func Berührt() -> Bool
    {
        MeineView.v.figurenLock.lock()
        let res = MeineView.v.figuren.first(where: {($0 !== self) && $0.Schneidet(wen: self.form) && $0.sichtbar && (!($0 is TurtleIntern) || ($0 as! TurtleIntern).symbolSichtbar)}) != nil
        MeineView.v.figurenLock.unlock()
        return res
    }

    /**
    Testet, ob die Figur eine (sichtbare) Grafik-Figur in der angegebenen Farbe berührt.
    - parameters:
        - farbe: die Farbe, die die berührte Figur haben muss
    - returns: true, wenn die Figurkoordinaten innerhalb einer Grafikfigur in der angegebenen Farbe sind
     */
    func Berührt(farbe: String) -> Bool
    {
        let color2 = FarbeCodieren(farbe: farbe)
        var ok = false
        MeineView.v.figurenLock.lock()
        for f in MeineView.v.figuren
        {
            if (f !== self) && f.sichtbar && f.Schneidet(wen: self.form)
            {
                if let t = f as? TurtleIntern
                {
                    if t.symbolSichtbar
                    {
                        for e in t.standardFigur
                        {
                            let rot = CGAffineTransform(translationX: t.x, y: t.y).rotated(by: DrehwinkelGeben(winkel: t.winkel))
                            let fg = CGMutablePath()
                            let größe = t.breite > t.höhe ? t.höhe : t.breite
                            e.ElementZuForm(p: fg, größe: größe, trans: rot)
                            if Schneidet(wen: fg) && (color2 == e.c)
                            {
                                ok = true;
                            }
                        }
                    }
                }
                else if let t = f as? FigurIntern
                {
                    let figur = t.eigeneFigur.count > 0 ? t.eigeneFigur : t.standardFigur
                    for e in figur
                    {
                        let rot = CGAffineTransform(translationX: t.x, y: t.y).rotated(by: DrehwinkelGeben(winkel: t.winkel))
                        let f = CGMutablePath()
                        let größe = t.breite > t.höhe ? t.höhe : t.breite
                        e.ElementZuForm(p: f, größe: größe, trans: rot)
                        if Schneidet(wen: f) && (color2 == e.c)
                        {
                            ok = true;
                        }
                    }
                }
                else
                {
                    ok = ok || (color2 == f.color)
                }
            }
        }
        MeineView.v.figurenLock.unlock()
        return ok
    }

    /**
    Testet, ob die Figur die (sichtbare, ) angegebene Grafik-Figur berührt.
    - parameters:
        - object: das Objekt, das getestet werden soll
    - returns: true, wenn die Turtlekoordinaten innerhalb einer Grafikfigur in der angegebenen Farbe sind
     */
    func Berührt(object: AnyObject) -> Bool
    {
        if let o = SymbolGeben(objekt: object)
        {
            return (o !== self) && o.Schneidet(wen: self.form) && o.sichtbar && (!(o is TurtleIntern) || (o as! TurtleIntern).symbolSichtbar)
        }
        else
        {
            return false
        }
    }
    
    /**
    Erzeugt ein neues, rechteckiges Element einer eigenen Darstellung der Figur.
 
    Alle Werte beziehen sich auf eine Figur der Größe 100 und den Koordinaten (0|0) in der Mitte des Quadrats
    - parameters:
        - x: x-Wert der linken oberen Ecke des Rechtecks
        - y: y-Wert der linken oberen Ecke des Rechtecks
        - breite: Breite des Rechtecks
        - höhe: Höhe des Rechtecks
        - farbe: (Füll)Farbe des Rechtecks
    */
    func FigurteilFestlegenRechteck (x: Int, y: Int, breite: Int, höhe: Int, farbe: String)
    {
        eigeneFigur.append(FigurenElementRechteck(xe: CGFloat(x), ye: CGFloat(y), breite: CGFloat(breite), höhe: CGFloat(höhe), c: FarbeCodieren(farbe: farbe)))
        FormErzeugen()
        MeineView.v.NeuZeichnen()
    }
    
    /**
    Erzeugt ein neues, elliptisches Element einer eigenen Darstellung der Figur.
 
    Alle Werte beziehen sich auf eine Figur der Größe 100 und den Koordinaten (0|0) in der Mitte des Quadrats
    - parameters:
        - x: x-Wert der linken oberen Ecke des umgebenden Rechtecks der Ellipse
        - y: y-Wert der linken oberen Ecke des umgebenden Rechtecks der Ellipse
        - breite: Breite des umgebenden Rechtecks der Ellipse
        - höhe: Höhe des umgebenden Rechtecks der Ellipse
        - farbe: (Füll)Farbe der Ellipse
    */
    func FigurteilFestlegenEllipse (x: Int, y: Int, breite: Int, höhe: Int, farbe: String)
    {
        eigeneFigur.append(FigurenElementEllipse(xe: CGFloat(x), ye: CGFloat(y), breite: CGFloat(breite), höhe: CGFloat(höhe), c: FarbeCodieren(farbe: farbe)))
        FormErzeugen()
        MeineView.v.NeuZeichnen()
    }
    
    /**
    Erzeugt ein neues, dreieckiges Element einer eigenen Darstellung der Figur.
 
    Alle Werte beziehen sich auf eine Figur der Größe 100 und den Koordinaten (0|0) in der Mitte des Quadrats
    - parameters:
        - x1: x-Wert des ersten Punkts des Dreiecks
        - y1: y-Wert des ersten Punkts des Dreiecks
        - x2: x-Wert des zweiten Punkts des Dreiecks
        - y2: y-Wert des zweiten Punkts des Dreiecks
        - x3: x-Wert des dritten Punkts des Dreiecks
        - y3: y-Wert des dritten Punkts des Dreiecks
        - farbe: (Füll)Farbe der Ellipse
    */
    func FigurteilFestlegenDreieck (x1: Int, y1: Int, x2: Int, y2: Int, x3: Int, y3: Int, farbe: String)
    {
        eigeneFigur.append(FigurenElementPolygon(p: [CGPoint(x: x1, y: y1), CGPoint(x: x2, y: y2), CGPoint(x: x3, y: y3)], c: FarbeCodieren(farbe: farbe)))
        FormErzeugen()
        MeineView.v.NeuZeichnen()
    }

    /**
    Löscht die Vereinbarung für die eigene Darstellung Figur.
    
    Die Figur wird wieder durch die Originalfigur dargestellt.
    */
    func EigeneFigurLöschen()
    {
        eigeneFigur.removeAll()
        FormErzeugen()
        MeineView.v.NeuZeichnen()
    }

    /**
    Zeichnet die Turtle mit ihrer Spur in den gegebenen Context.
    - parameters:
        - context: Grafikkontext
    */
    override func Zeichnen(context: CGContext)
    {
        // Outline
        context.saveGState()
        context.setStrokeColor(SystemFarbe.black.cgColor)
        context.setLineWidth(3.0)
        context.beginPath()
        context.addPath(form)
        context.strokePath()
        // Füllung
        let größe = breite > höhe ? höhe : breite
        context.translateBy(x: x, y: y)
        context.rotate(by: DrehwinkelGeben (winkel: winkel))
        let figur = eigeneFigur.count > 0 ? eigeneFigur : standardFigur
        for e in figur
        {
            e.ElementZeichnen(context: context, größe: größe)
        }
        context.restoreGState()
    }
}

/**
stellt einen Zufallsgenerator zur Verfügung.
*/
public class Zufall
{
    /**
    Initialisiert die verwendete Instanz des System-Zufallsgenerators
    */
    public init ()
    {
        srandom(0)
    }
    
    /**
    Typische Kommazufallszahl
    - returns: Eine Zufallszahl aus dem Intervall [0; 1[
    */
    public func ZufallKomma () -> Float
    {
        return Float (arc4random()) / Float (UINT32_MAX)
    }

    /**
    Typische ganze Zufallszahl
    - returns: Eine ganze Zufallszahl aus dem Intervall [0; top[
    */
    public func ZufallGanz (top: Int) -> Int
    {
        return Int (ZufallKomma() * Float (top))
    }
}

/**
Methoden für den Aufruf beim Mausklick / Touch
*/
protocol AktionsEmpfänger: AnyObject
{
    func AktionAusführen()
    func TasteGedrückt(taste: Character)
    func SonderTasteGedrückt(taste: Int)
    func MausGeklickt(x:Int, y: Int, anzahl: Int)
}

/**
Die Rahmenview für die Darstellung der Graphikobjekte.
*/
class MeineView: View
{
	/** Aufhänger für "die" View. */
    static var v: MeineView! = nil
    /** Letzter Berührpunkt, (-10 | -10) für keine Berührung */
    var point: CGPoint
    /** Feld der zu zeichnenden Figuren */
    var figuren: [ZeichenObjekt]
    /** Feld aller Aktionsempfänger. */
    var aktionsEmpfänger: [AktionsEmpfänger]
    /** Referenz auf das Timer-Objekt */
    private var timer: Timer!
    /** Aktuell eingestelltes Zeitintervall in Sekunden */
    private var interval: TimeInterval
    /** Lock für das Feld figuren */
    var figurenLock: NSLock
    /** Lock für das Feld aktionsEmpfänger */
    private var aktionenLock: NSLock
    #if os(macOS)
    /** Hintergrundfarbe */
    var backgroundColor: SystemFarbe
    override var acceptsFirstResponder: Bool {get {return true}}
    #elseif os(iOS)
	/** Aufhänger für "die" View. */
    static var g: GesamtFenster! = nil
    #endif

    /**
    Der Konstruktor setzt die Hintergrundfarbe.
    - parameters:
        - frame: Der zur Verfügung gestellte Platz
    */
    override init (frame: CGRect)
    {
        figuren = [ZeichenObjekt] ()
        figurenLock = NSLock()
        aktionsEmpfänger = [AktionsEmpfänger]()
        aktionenLock = NSLock()
        point = CGPoint(x: -10, y: -10)
        timer = nil
        interval = 1.0
        #if os(macOS)
        backgroundColor = SystemFarbe (red: 0.9, green: 0.9, blue: 0.9, alpha: 1.0)
        #endif
        super.init(frame: frame)
        #if os(iOS)
        backgroundColor = SystemFarbe (red: 0.9, green: 0.9, blue: 0.9, alpha: 1.0)
        #endif
    }

    /**
    Der alternative Konstruktor setzt die Hintergrundfarbe.
    - parameters:
        - coder: Das zur Verfügung gestellte Objekt der Klasse NSCoder
    */
    required init? (coder: NSCoder)
    {
        figuren = [ZeichenObjekt] ()
        figurenLock = NSLock()
        aktionsEmpfänger = [AktionsEmpfänger]()
        aktionenLock = NSLock()
        point = CGPoint(x: -10, y: -10)
        timer = nil
        interval = 1.0
        #if os(macOS)
        backgroundColor = SystemFarbe (red: 0.9, green: 0.9, blue: 0.9, alpha: 1.0)
        #endif
        super.init (coder: coder)
        #if os(iOS)
        backgroundColor = SystemFarbe (red: 0.9, green: 0.9, blue: 0.9, alpha: 1.0)
        #endif
    }
    
    /**
    Interne Funktion, um den Parameter von TaktImpulsAusführen fernzuhalten
    - parameters:
        - t: der aufrufende Timer
    */
    @objc func Impuls (t: Timer)
    {
        for e in aktionsEmpfänger
        {
            e.AktionAusführen()
        }
    }

    /**
    Startet den Timer mit der eingestellten Taktlänge.
    */
    func Start ()
    {
        if timer != nil
        {
            timer.invalidate();
        }
        timer = Timer.scheduledTimer(timeInterval: interval, target: self, selector: #selector(MeineView.Impuls), userInfo: nil, repeats: true)
    }

    /**
    Stopt den Timer.
    */
    func Stop ()
    {
        if timer != nil
        {
            timer.invalidate();
            timer = nil
        }
    }

    /**
    Stellt die Taktlänge des Timers ein.
    - parameters:
        - s: Taktlänge in Sekunden; beliebige Bruchteile sind möglich.
    */
    func TaktlängeSetzen (s: Double)
    {
        if timer != nil
        {
            Stop()
            interval = s
            Start()
        }
        else
        {
            interval = s
        }
    }

	/**
	Trägt einen neuen Aktionsempfänger ein.
	- parameters:
		- neu: der neue Aktionsempfänger
	*/
	func AktionsEmpfängerEintragen(neu: AktionsEmpfänger)
	{
		aktionsEmpfänger.append(neu)
    }

    /**
    Löscht einen Aktionsempfänger aus der Liste.
    - parameters:
        - alt: der zu löschende Aktionsempfänger
    */
    func AktionsEmpfängerEntfernen(alt: AktionsEmpfänger)
    {
        aktionsEmpfänger.remove(at: aktionsEmpfänger.firstIndex(where: {$0 === alt})!)
    }

    /**
    Die Zeichenmethode wird immer dann aufgrufen, wenn edie View oder ein Teil davon neu gezeichnet werden muss.
    - parameters:
        - rect: das (mindestens) neu zu zeichnende Rechteck
    */
    override func draw(_ rect: CGRect)
    {
        #if os(macOS)
        if let context = NSGraphicsContext.current?.cgContext
        {
            context.scaleBy(x:1, y:-1)
            context.translateBy(x:0, y:-self.bounds.height)
            context.setFillColor(backgroundColor.cgColor)
            context.fill(bounds)
            context.textMatrix  = CGAffineTransform(a: 1, b: 0.0, c: 0.0, d: -1, tx: 0.0, ty: 0.0)
            figurenLock.lock()
            for figur in figuren
            {
                if figur.sichtbar
                {
                    figur.Zeichnen (context: context)
                }
            }
            figurenLock.unlock()
        }
        #elseif os(iOS)
        if let context = UIGraphicsGetCurrentContext()
        {
            context.textMatrix  = CGAffineTransform(a: 1, b: 0.0, c: 0.0, d: -1, tx: 0.0, ty: 0.0)
            figurenLock.lock()
            for figur in figuren
            {
                if figur.sichtbar
                {
                    figur.Zeichnen (context: context)
                }
            }
            figurenLock.unlock()
        }
        #endif
    }
    
    #if os(macOS)
    /**
    Aufgerufen, wenn die Maus gedrückt wird.
    - parameters:
        - event: das Ereignis, zu dem der Klick gehört
    */
    override func mouseDown(with event: NSEvent)
    {
        point = convert(event.locationInWindow, from: nil)
    }

    /**
    Aufgerufen, wenn die Maus losgelassen wird.
    - parameters:
        - event: das Ereignis, zu dem der Klick gehört
    */
    override func mouseUp(with event: NSEvent)
    {
        let p = convert(event.locationInWindow, from: nil)
        let r = CGRect(origin: point, size: CGSize(width: 10, height: 10)).offsetBy(dx: -5, dy: -5)
        if r.contains(p)
        {
            for e in aktionsEmpfänger
            {
            	e.MausGeklickt(x: Int(p.x), y: Int(p.y), anzahl: event.clickCount)
            }
        }
        point = CGPoint(x: -10, y: -10)
    }
    
    /**
    Aufgerufen, wenn eine Taste gedrückt wird.
    - parameters:
        - event: das Ereignis, zu dem der Tastendruck gehört
    */
    override func keyDown(with event: NSEvent)
    {
        for e in aktionsEmpfänger
        {
            if event.characters!.first!.hexDigitValue != nil
            {
            	e.TasteGedrückt(taste: event.characters!.first!)
            }
            else
            {
                e.SonderTasteGedrückt(taste: Int(event.keyCode))
            }
        }
    }

    #elseif os(iOS)
    /**
    Aufgerufen, wenn ein oder mehr Finger die Tastatur berühren.
    - parameters:
        - touches: Menge an Berüchrungsbeschreibungen für den Beginn des Ereignisses
        - event: das Gesamtereignis, zu dem die Berührungen gehören
    */
    override func touchesBegan(_ touches: Set<UITouch>, with event: UIEvent?) {
        for t in touches as Set<UITouch>
        {
            point = t.location(in: self)
        }
        super.touchesBegan(touches, with: event)
    }

    /**
    Aufgerufen, wenn ein oder mehr Finger die Tastatur verlassen.
    - parameters:
        - touches: Menge an Berührungsbeschreibungen für den Beginn des Ereignisses
        - event: das Gesamtereignis, zu dem die Berührungen gehören
    */
    override func touchesEnded(_ touches: Set<UITouch>, with event: UIEvent?)
    {
        for t in touches as Set<UITouch>
        {
            let point2 = t.location(in: self)
            let r = CGRect(origin: point, size: CGSize(width: 10, height: 10)).offsetBy(dx: -5, dy: -5)
            if r.contains(point2)
            {
                for e in aktionsEmpfänger
                {
                    e.MausGeklickt(x: Int(point2.x), y: Int(point2.y), anzahl: t.tapCount)
                }
            }
        }
        point = CGPoint(x: -10, y: -10)
        super.touchesEnded(touches, with: event)
    }

    override func tintColorDidChange() {
        setNeedsDisplay()
    }
    #endif

    func NeuZeichnen()
    {
        #if os(macOS)
        needsDisplay = true
        #elseif os(iOS)
        setNeedsDisplay()
        #endif
    }
}

#if os(macOS)
/**
Gesamtfenster (View + Tasten/Regler)
*/
class GesamtFenster: NSView
{
    /** die eigentliche Anzeigeview */
    private var anzeige: MeineView! = nil
    /** Referenz auf den Start-Knopf */
    private var startKnopf: NSButton! = nil
    /** Referenz auf den Stop-Knopf */
    private var stopKnopf: NSButton! = nil
    /** Slider für die Taktgeschwindigkeit */
    private var slider: NSSlider! = nil
    

    /*
    Standardkonstruktor
    */
    override init (frame: CGRect)
    {
        super.init(frame: frame)
        ElementeAnlegen ()
    }

    /*
    Konstruktor für Erstellung aus NIB-File
    */
    required init? (coder: NSCoder)
    {
        super.init (coder: coder)
        ElementeAnlegen ()
    }
    
    /**
    Aktionsfunktion für den Start-Knopf
    - parameters:
        - g: das implementierende Gesamtfenster
    */
    @objc func TimerStart (g: GesamtFenster)
    {
        anzeige.Start()
        startKnopf.isEnabled = false
        stopKnopf.isEnabled = true
    }
    
    /**
    Aktionsfunktion für den Stop-Knopf
    - parameters:
        - g: das implementierende Gesamtfenster
    */
    @objc func TimerStop (g: GesamtFenster)
    {
        anzeige.Stop()
        startKnopf.isEnabled = true
        stopKnopf.isEnabled = false
    }
    
    /**
    Aktionsfunktion für die Taktzeiteinstellung
    - parameters:
        - g: das implementierende Gesamtfenster
    */
    @objc func TimerWert (g: GesamtFenster)
    {
        anzeige.TaktlängeSetzen(s: slider.doubleValue / 1000)
    }

    /**
    Legt die Oberflächenelemente an
    */
    private func ElementeAnlegen ()
    {
        anzeige = MeineView(frame: CGRect(x: 0.0, y: 30.0, width: anzeigebreite, height: anzeigeHöhe))
        addSubview(anzeige)
        MeineView.v = anzeige
        startKnopf = NSButton(title: "Start", target: self, action: #selector(GesamtFenster.TimerStart))
        startKnopf.setFrameOrigin(NSPoint(x: 0, y: 0))
        startKnopf.setFrameSize(NSSize(width: 80, height: 30))
        addSubview(startKnopf)
        stopKnopf = NSButton(title: "Stop", target: self, action: #selector(GesamtFenster.TimerStop))
        stopKnopf.setFrameOrigin(NSPoint(x: 90, y: 0))
        stopKnopf.setFrameSize(NSSize(width: 80, height: 30))
        stopKnopf.isEnabled = false
        addSubview(stopKnopf)
        slider = NSSlider(value: 500, minValue: 0, maxValue: 1000, target: self, action: #selector(GesamtFenster.TimerWert))
        slider.numberOfTickMarks = 10
        slider.setFrameOrigin(NSPoint(x: 180, y: 0))
        slider.setFrameSize(NSSize(width: 200, height: 30))
        addSubview(slider)
    }
}
#elseif os(iOS)
/**
Gesamtfenster (View + Tasten/Regler)
*/
public class GesamtFenster: UIView
{
    /** die eigentliche Anzeigeview */
    private var anzeige: MeineView! = nil
    /** Referenz auf den Start-Knopf */
    private var startKnopf: UIButton! = nil
    /** Referenz auf den Stop-Knopf */
    private var stopKnopf: UIButton! = nil
    /** Slider für die Taktgeschwindigkeit */
    private var slider: UISlider! = nil
    /** Links-Pfeil */
    private var linksKnopf: UIButton! = nil
    /** Rechts-Pfeil */
    private var rechtsKnopf: UIButton! = nil
    /** Oben-Pfeil */
    private var obenKnopf: UIButton! = nil
    /** Unten-Pfeil */
    private var untenKnopf: UIButton! = nil
    /** Buchstabentasten */
    private var keys: [UIButton]! = nil
    /** Referenz auf das Gesamtfenster */
    private static var g: GesamtFenster? = nil

    /*
    Standardkonstruktor
    */
    override init (frame: CGRect)
    {
        super.init(frame: frame)
        backgroundColor = UIColor.white
        ElementeAnlegen ()
        GesamtFenster.g = self
    }

    /*
    Konstruktor für Erstellung aus NIB-File
    */
    required init? (coder: NSCoder)
    {
        super.init (coder: coder)
        ElementeAnlegen ()
        GesamtFenster.g = self
    }
    
    /**
	Setzt den Titel und den Rückgabewert für den Knopf mit der angegebenen Nummer
	- parameters:
		- nr: die Nummer des Knopfs (0 <= nr < 10)
        - c: der Titelbuchstabe
     */
    public class func KnopfbuchstabenSetzen (nr: Int, c: Character)
    {
    	if let g2 = GesamtFenster.g
    	{
    		if (nr >= 0) && (nr < 10)
        	{
                g2.keys[nr].setTitle(String(c), for: .normal)
            }
        }
    }
    
    /**
	Update für Slider
    */
    func SliderSetzen (wert: Int)
    {
    	slider!.value = Float(wert)
	}
    
    /**
    Aktionsfunktion für den Start-Knopf
    - parameters:
        - g: das implementierende Gesamtfenster
    */
    @objc func TimerStart (g: GesamtFenster)
    {
        anzeige.Start()
        startKnopf.isEnabled = false
        stopKnopf.isEnabled = true
    }
    
    /**
    Aktionsfunktion für den Stop-Knopf
    - parameters:
        - g: das implementierende Gesamtfenster
    */
    @objc func TimerStop (g: GesamtFenster)
    {
        anzeige.Stop()
        startKnopf.isEnabled = true
        stopKnopf.isEnabled = false
    }
    
    /**
    Aktionsfunktion für die Taktzeiteinstellung
    - parameters:
        - g: das implementierende Gesamtfenster
    */
    @objc func TimerWert (g: GesamtFenster)
    {
        anzeige.TaktlängeSetzen(s: Double(slider!.value / 1000))
    }
    
    /**
    Aktionsfunktion für den Linkspfeil
    - parameters:
        - g: das implementierende Gesamtfenster
    */
    @objc func LinksKnopfGedrückt (g: GesamtFenster)
    {
        for e in anzeige.aktionsEmpfänger
        {
            e.SonderTasteGedrückt(taste: 123)
        }

    }
    
    /**
    Aktionsfunktion für den Rechtspfeil
    - parameters:
        - g: das implementierende Gesamtfenster
    */
    @objc func RechtsKnopfGedrückt (g: GesamtFenster)
    {
        for e in anzeige.aktionsEmpfänger
        {
            e.SonderTasteGedrückt(taste: 124)
        }

    }
    
    /**
    Aktionsfunktion für den Obenpfeil
    - parameters:
        - g: das implementierende Gesamtfenster
    */
    @objc func ObenKnopfGedrückt (g: GesamtFenster)
    {
        for e in anzeige.aktionsEmpfänger
        {
            e.SonderTasteGedrückt(taste: 126)
        }

    }
    
    /**
    Aktionsfunktion für den Untenpfeil
    - parameters:
        - g: das implementierende Gesamtfenster
    */
    @objc func UntenKnopfGedrückt (g: GesamtFenster)
    {
        for e in anzeige.aktionsEmpfänger
        {
            e.SonderTasteGedrückt(taste: 125)
        }

    }
    
    /**
    Aktionsfunktion für Buchstabenknöpfe
    - parameters:
        - g: das implementierende Gesamtfenster
    */
    @objc func BuchstabenKnopfGedrückt (sender: UIButton)
    {
        for e in anzeige.aktionsEmpfänger
        {
            e.TasteGedrückt(taste: sender.currentTitle!.first!)
        }

    }

    /**
    Legt die Oberflächenelemente an
    */
    private func ElementeAnlegen ()
    {
        anzeige = MeineView(frame: CGRect(x: 0.0, y: 0.0, width: anzeigebreite, height: anzeigeHöhe))
        addSubview(anzeige)
        MeineView.v = anzeige
        MeineView.g = self
        startKnopf = UIButton(type: .roundedRect)
        startKnopf.addTarget(self, action: #selector(GesamtFenster.TimerStart), for: .touchUpInside)
        startKnopf.frame = CGRect(origin: CGPoint(x: 0, y: anzeigeHöhe), size: CGSize(width: 80, height: 30))
        startKnopf.setTitle("Start", for: .normal)
        startKnopf.isEnabled = true
        addSubview(startKnopf)
        stopKnopf = UIButton(type: .roundedRect)
        stopKnopf.addTarget(self, action: #selector(GesamtFenster.TimerStop), for: .touchUpInside)
        stopKnopf.frame = CGRect(origin: CGPoint(x: 90, y: anzeigeHöhe), size: CGSize(width: 80, height: 30))
        stopKnopf.setTitle("Stop", for: .normal)
        stopKnopf.isEnabled = false
        addSubview(stopKnopf)
        slider = UISlider(frame: CGRect(origin: CGPoint(x: 180, y: anzeigeHöhe), size: CGSize(width: 200, height: 30)))
        slider.minimumValue = 0
        slider.maximumValue = 1000
        slider.value = 500
        slider.addTarget(self, action: #selector(GesamtFenster.TimerWert), for: .valueChanged)
        slider.isContinuous = false
        addSubview(slider)
        linksKnopf = UIButton(type: .roundedRect)
        linksKnopf.addTarget(self, action: #selector(GesamtFenster.LinksKnopfGedrückt), for: .touchUpInside)
        linksKnopf.frame = CGRect(origin: CGPoint(x: 0, y: anzeigeHöhe + 30), size: CGSize(width: 40, height: 30))
        linksKnopf.setTitle("←", for: .normal)
        linksKnopf.isEnabled = true
        addSubview(linksKnopf)
        untenKnopf = UIButton(type: .roundedRect)
        untenKnopf.addTarget(self, action: #selector(GesamtFenster.UntenKnopfGedrückt), for: .touchUpInside)
        untenKnopf.frame = CGRect(origin: CGPoint(x: 50, y: anzeigeHöhe + 30), size: CGSize(width: 40, height: 30))
        untenKnopf.setTitle("↓", for: .normal)
        untenKnopf.isEnabled = true
        addSubview(untenKnopf)
        obenKnopf = UIButton(type: .roundedRect)
        obenKnopf.addTarget(self, action: #selector(GesamtFenster.ObenKnopfGedrückt), for: .touchUpInside)
        obenKnopf.frame = CGRect(origin: CGPoint(x: 100, y: anzeigeHöhe + 30), size: CGSize(width: 40, height: 30))
        obenKnopf.setTitle("↑", for: .normal)
        obenKnopf.isEnabled = true
        addSubview(obenKnopf)
        rechtsKnopf = UIButton(type: .roundedRect)
        rechtsKnopf.addTarget(self, action: #selector(GesamtFenster.RechtsKnopfGedrückt), for: .touchUpInside)
        rechtsKnopf.frame = CGRect(origin: CGPoint(x: 150, y: anzeigeHöhe + 30), size: CGSize(width: 40, height: 30))
        rechtsKnopf.setTitle("→", for: .normal)
        rechtsKnopf.isEnabled = true
        addSubview(rechtsKnopf)
        let z: [String] = ["a", "s", "d", "w", "x", "j", "k", "l", "i", "m"]
        keys = []
        for nr in 0 ..< 10
        {
        	keys.append(UIButton(type: .roundedRect))
        	keys[nr].addTarget(self, action: #selector(GesamtFenster.BuchstabenKnopfGedrückt), for: .touchUpInside)
            keys[nr].frame = CGRect(origin: CGPoint(x: 200.0 + Double(nr) * 50, y: anzeigeHöhe + 30), size: CGSize(width: 40, height: 30))
            keys[nr].setTitle(z[nr], for: .normal)
            keys[nr].isEnabled = true
            addSubview(keys[nr])
        }
    }
}
#endif

/**
Wrapperklasse für ein Rechteck auf der Zeichenfläche.
*/
open class Rechteck
{
    /** x-Position der linken oberen Ecke. */
    public var x: Int
    /** y-Position der linken oberen Ecke. */
    public var y: Int
    /** Breite des Rechtecks. */
    public var breite: Int
    /** Höhe des Rechtecks. */
    public var höhe: Int
    /** Farbe des Rechtecks. */
    public var farbe: String
    /** Sichtbarkeit des Rechtecks. */
    public var sichtbar: Bool
    /** Drehwinkel (mathematisch positiver Drehsinn) des Rechtecks in Grad. */
    public var winkel: Int
    /** Referenz auf das Delegate-Objekt. */
    var symbol: ZeichenObjekt
    
    /**
    Der Konstruktor erzeugt das Delegate-Objekt
    */
    public init()
    {
        x = 10
        y = 10
        breite = 100
        höhe = 100
        farbe = "rot"
        sichtbar = true
        winkel = 0
        symbol = RechteckIntern()
        symbol.PositionSetzen(x: x, y: y)
        symbol.GrößeSetzen(breite: breite, höhe: höhe)
        symbol.FarbeSetzen(farbe: farbe)
        symbol.SichtbarkeitSetzen(sichtbar: sichtbar)
        symbol.WinkelSetzen(winkel: winkel)
    }
    
    /**
    Setzt die Position (der linken oberen Ecke) des Rechtecks.
    - parameters:
        - x: x-Position der linken oberen Ecke
        - y: y-Position der linken oberen Ecke
    */
    open func PositionSetzen (x: Int, y: Int)
    {
        self.x = x
        self.y = y
        symbol.PositionSetzen(x: x, y: y)
    }
    
    /**
    Verschiebt das Rechteck um die angegebenen Werte.
    - parameters:
        - deltaX: Verschiebung in x-Richtung
        - deltaY: Verschiebung in y-Richtung
    */
    open func Verschieben (deltaX: Int, deltaY: Int)
    {
        x += deltaX
        y += deltaY
        symbol.PositionSetzen(x: x, y: y)
    }
    
    /**
    Dreht das Rechteck
    - parameters:
        - grad: Drehwinkel (mathematisch positiver Drehsinn) im Gradmass
    */
    open func Drehen (grad: Int)
    {
        winkel += grad
        symbol.WinkelSetzen(winkel: winkel)
    }
    
    /**
    Setzt die Größe des Rechtecks.
    - parameters:
        - breite: (neue) Breite
        - höhe: (neue) Höhe
    */
    open func GrößeSetzen (breite: Int, höhe: Int)
    {
        self.breite = breite
        self.höhe = höhe
        symbol.GrößeSetzen(breite: breite, höhe: höhe)
    }
    
    /**
    Setzt die Farbe des Rechtecks.
    Erlaubte Farben sind:
    "weiß", "weiss", "rot", "grün", "gruen", "blau", "gelb",
    "magenta", "cyan", "hellgelb", "hellgrün", "hellgruen",
    "orange", "braun", "grau", "schwarz"
    Alle anderen Eingaben werden auf die Farbe schwarz abgebildet.
    - parameters:
        - farbe: (neue) Farbe
    */
    open func FarbeSetzen (farbe: String)
    {
        self.farbe = farbe
        symbol.FarbeSetzen(farbe: farbe)
    }
    
    /**
    Setzt den Drehwinkel des Rechtecks.
    Die Winkelangabe ist in Grad,positive Werte drehen gegen den Uhrzeigersinn,
    negative Werte drehen im Uhrzeigersinn (mathematisch positiver Drehsinn).
    - parameters:
        - winkel: der (neue) Drehwinkel des Rechtecks
    */
    open func WinkelSetzen (winkel: Int)
    {
        self.winkel = winkel
        symbol.WinkelSetzen(winkel: winkel)
    }
    
    /**
    Schaltet die Sichtbarkeit des Rechtecks ein oder aus.
    Erlaubte Parameterwerte: true, false
    - parameters:
        - sichtbar: (neue) Sichtbarkeit des Rechtecks
    */
    open func SichtbarkeitSetzen (sichtbar: Bool)
    {
        self.sichtbar = sichtbar
        symbol.SichtbarkeitSetzen(sichtbar: sichtbar)
    }
    
    /**
    Entfernt das Rechteck aus dem Zeichenfenster.
    */
    open func Entfernen ()
    {
        symbol.Entfernen()
    }

    /**
    Bringt das Rechteck eine Ebene nach vorn.
    */
    open func NachVornBringen()
    {
        symbol.NachVornBringen()
    }
    
    /**
    Bringt das Rechteck in die vorderste Ebene.
    */
    open func GanzNachVornBringen()
    {
        symbol.GanzNachVornBringen()
    }
    
    /**
    Bringt das Rechteck eine Ebene nach hinten.
    */
    open func NachHintenBringen()
    {
        symbol.NachHintenBringen()
    }
    
    /**
    Bringt das Rechteck in die hinterste Ebene.
    */
    open func GanzNachHintenBringen()
    {
        symbol.GanzNachHintenBringen()
    }
}

/**
Wrapperklasse für einen Kreis auf der Zeichenfläche.
*/
open class Kreis
{
    /** x-Position des Kreismittelpunktes. */
    public var x: Int
    /** y-Position des Kreismittelpunktes. */
    public var y: Int
    /** Radius des Kreises. */
    public var radius: Int
    /** Farbe des Rechtecks. */
    public var farbe: String
    /** Sichtbarkeit des Rechtecks. */
    public var sichtbar: Bool
    /** Drehwinkel (mathematisch positiver Drehsinn) des Rechtecks in Grad. */
    public var winkel: Int
    /** Referenz auf das Delegate-Objekt. */
    var symbol: ZeichenObjekt
    
    /**
    Der Konstruktor erzeugt das Delegate-Objekt
    */
    public init()
    {
        x = 60
        y = 60
        radius = 50
        farbe = "rot"
        sichtbar = true
        winkel = 0
        symbol = EllipseIntern()
        symbol.PositionSetzen(x: x - radius, y: y - radius)
        symbol.GrößeSetzen(breite: radius*2, höhe: radius*2)
        symbol.FarbeSetzen(farbe: farbe)
        symbol.SichtbarkeitSetzen(sichtbar: sichtbar)
        symbol.WinkelSetzen(winkel: winkel)
    }
    
    /**
    Setzt die Position (des Mittelpunkts) des Kreises.
    - parameters:
        - x: x-Position des Mittelpunkts
        - y: y-Position des Mittelpunkts
    */
    open func PositionSetzen (x: Int, y: Int)
    {
        self.x = x
        self.y = y
        symbol.PositionSetzen(x: x - radius, y: y - radius)
    }
    
    /**
    Verschiebt den Kreis um die angegebenen Werte.
    - parameters:
        - deltaX: Verschiebung in x-Richtung
        - deltaY: Verschiebung in y-Richtung
    */
    open func Verschieben (deltaX: Int, deltaY: Int)
    {
        x += deltaX
        y += deltaY
        symbol.PositionSetzen(x: x - radius, y: y - radius)
    }
    
    /**
    Dreht den Kreis
    - parameters:
        - grad: Drehwinkel (mathematisch positiver Drehsinn) im Gradmass
    */
    open func Drehen (grad: Int)
    {
        winkel += grad
        symbol.WinkelSetzen(winkel: winkel)
    }
    
    /**
    Setzt den Radius des Kreises.
    - parameters:
        - radius: (neuer) Radius
    */
    open func RadiusSetzen (radius: Int)
    {
        self.radius = radius
        symbol.GrößeSetzen(breite: radius*2, höhe: radius*2)
        symbol.PositionSetzen(x: x - radius, y: y - radius)
    }
    
    /**
    Setzt die Farbe des Kreises.
    Erlaubte Farben sind:
    "weiß", "weiss", "rot", "grün", "gruen", "blau", "gelb",
    "magenta", "cyan", "hellgelb", "hellgrün", "hellgruen",
    "orange", "braun", "grau", "schwarz"
    Alle anderen Eingaben werden auf die Farbe schwarz abgebildet.
    - parameters:
        - farbe: (neue) Farbe
    */
    open func FarbeSetzen (farbe: String)
    {
        self.farbe = farbe
        symbol.FarbeSetzen(farbe: farbe)
    }
    
    /**
    Setzt den Drehwinkel des Kreises.
    Die Winkelangabe ist in Grad,positive Werte drehen gegen den Uhrzeigersinn,
    negative Werte drehen im Uhrzeigersinn (mathematisch positiver Drehsinn).
    - parameters:
        - winkel: der (neue) Drehwinkel des Kreises
    */
    open func WinkelSetzen (winkel: Int)
    {
        self.winkel = winkel
        symbol.WinkelSetzen(winkel: winkel)
    }
    
    /**
    Schaltet die Sichtbarkeit des Kreises ein oder aus.
    Erlaubte Parameterwerte: true, false
    - parameters:
        - sichtbar: (neue) Sichtbarkeit des Kreises
    */
    open func SichtbarkeitSetzen (sichtbar: Bool)
    {
        self.sichtbar = sichtbar
        symbol.SichtbarkeitSetzen(sichtbar: sichtbar)
    }
    
    /**
    Entfernt den Kreis aus dem Zeichenfenster.
    */
    open func Entfernen ()
    {
        symbol.Entfernen()
    }

    /**
    Bringt den Kreis eine Ebene nach vorn.
    */
    open func NachVornBringen()
    {
        symbol.NachVornBringen()
    }
    
    /**
    Bringt den Kreis in die vorderste Ebene.
    */
    open func GanzNachVornBringen()
    {
        symbol.GanzNachVornBringen()
    }
    
    /**
    Bringt den Kreis eine Ebene nach hinten.
    */
    open func NachHintenBringen()
    {
        symbol.NachHintenBringen()
    }
    
    /**
    Bringt den Kreis in die hinterste Ebene.
    */
    open func GanzNachHintenBringen()
    {
        symbol.GanzNachHintenBringen()
    }
}

/**
Wrapperklasse für ein Dreieck auf der Zeichenfläche.
*/
open class Dreieck
{
    /** x-Position der Spitze. */
    public var x: Int
    /** y-Position der Spitze. */
    public var y: Int
    /** Breite des umgebenden Rechtecks. */
    public var breite: Int
    /** Höhe des umgebenden Rechtecks. */
    public var höhe: Int
    /** Farbe des Dreiecks. */
    public var farbe: String
    /** Sichtbarkeit des Dreiecks. */
    public var sichtbar: Bool
    /** Drehwinkel (mathematisch positiver Drehsinn) des Dreiecks in Grad. */
    public var winkel: Int
    /** Referenz auf das Delegate-Objekt. */
    var symbol: ZeichenObjekt
    
    /**
    Der Konstruktor erzeugt das Delegate-Objekt
    */
    public init()
    {
        x = 60
        y = 10
        breite = 100
        höhe = 100
        farbe = "rot"
        sichtbar = true
        winkel = 0
        symbol = DreieckIntern()
        symbol.PositionSetzen(x: x - breite/2, y: y)
        symbol.GrößeSetzen(breite: breite, höhe: höhe)
        symbol.FarbeSetzen(farbe: farbe)
        symbol.SichtbarkeitSetzen(sichtbar: sichtbar)
        symbol.WinkelSetzen(winkel: winkel)
    }
    
    /**
    Setzt die Position (der Spitze) des Dreiecks.
    - parameters:
        - x: x-Position der Spitze
        - y: y-Position der Spitze
    */
    open func PositionSetzen (x: Int, y: Int)
    {
        self.x = x
        self.y = y
        symbol.PositionSetzen(x: x - breite/2, y: y)
    }
    
    /**
    Verschiebt das Dreieck um die angegebenen Werte.
    - parameters:
        - deltaX: Verschiebung in x-Richtung
        - deltaY: Verschiebung in y-Richtung
    */
    open func Verschieben (deltaX: Int, deltaY: Int)
    {
        x += deltaX
        y += deltaY
        symbol.PositionSetzen(x: x - breite/2, y: y)
    }
    
    /**
    Dreht das Dreieck
    - parameters:
        - grad: Drehwinkel (mathematisch positiver Drehsinn) im Gradmass
    */
    open func Drehen (grad: Int)
    {
        winkel += grad
        symbol.WinkelSetzen(winkel: winkel)
    }
    
    /**
    Setzt die Größe des Dreiecks.
    - parameters:
        - breite: (neue) Breite
        - höhe: (neue) Höhe
    */
    open func GrößeSetzen (breite: Int, höhe: Int)
    {
        self.breite = breite
        self.höhe = höhe
        symbol.GrößeSetzen(breite: breite, höhe: höhe)
        symbol.PositionSetzen(x: x - breite/2, y: y)
    }
    
    /**
    Setzt die Farbe des Dreiecks.
    Erlaubte Farben sind:
    "weiß", "weiss", "rot", "grün", "gruen", "blau", "gelb",
    "magenta", "cyan", "hellgelb", "hellgrün", "hellgruen",
    "orange", "braun", "grau", "schwarz"
    Alle anderen Eingaben werden auf die Farbe schwarz abgebildet.
    - parameters:
        - farbe: (neue) Farbe
    */
    open func FarbeSetzen (farbe: String)
    {
        self.farbe = farbe
        symbol.FarbeSetzen(farbe: farbe)
    }
    
    /**
    Setzt den Drehwinkel des Dreiecks.
    Die Winkelangabe ist in Grad,positive Werte drehen gegen den Uhrzeigersinn,
    negative Werte drehen im Uhrzeigersinn (mathematisch positiver Drehsinn).
    - parameters:
        - winkel: der (neue) Drehwinkel des Dreiecks
    */
    open func WinkelSetzen (winkel: Int)
    {
        self.winkel = winkel
        symbol.WinkelSetzen(winkel: winkel)
    }
    
    /**
    Schaltet die Sichtbarkeit des Dreiecks ein oder aus.
    Erlaubte Parameterwerte: true, false
    - parameters:
        - sichtbar: (neue) Sichtbarkeit des Dreiecks
    */
    open func SichtbarkeitSetzen (sichtbar: Bool)
    {
        self.sichtbar = sichtbar
        symbol.SichtbarkeitSetzen(sichtbar: sichtbar)
    }
    
    /**
    Entfernt das Dreieck aus dem Zeichenfenster.
    */
    open func Entfernen ()
    {
        symbol.Entfernen()
    }

    /**
    Bringt das Dreieck eine Ebene nach vorn.
    */
    open func NachVornBringen()
    {
        symbol.NachVornBringen()
    }
    
    /**
    Bringt das Dreieck in die vorderste Ebene.
    */
    open func GanzNachVornBringen()
    {
        symbol.GanzNachVornBringen()
    }
    
    /**
    Bringt das Dreieck eine Ebene nach hinten.
    */
    open func NachHintenBringen()
    {
        symbol.NachHintenBringen()
    }
    
    /**
    Bringt das Dreieck in die hinterste Ebene.
    */
    open func GanzNachHintenBringen()
    {
        symbol.GanzNachHintenBringen()
    }
}

/**
Wrapperklasse für einen Text auf der Zeichenfläche.
*/
open class Text
{
    /** x-Position der linken Seite der Grundlinie. */
    private var x: Int
    /** y-Position der Grundlinie. */
    private var y: Int
    /** Farbe des Textes. */
    private var farbe: String
    /** Sichtbarkeit des Dreiecks. */
    private var sichtbar: Bool
    /** Drehwinkel (mathematisch positiver Drehsinn) des Dreiecks in Grad. */
    private var winkel: Int
    /** Größe des Textes in Punkten. */
    private var textgröße: Int
    /** Referenz auf das Delegate-Objekt. */
    var symbol: TextIntern
    
    /**
    Der Konstruktor erzeugt das Delegate-Objekt
    */
    public init()
    {
        x = 10
        y = 10
        farbe = "schwarz"
        sichtbar = true
        winkel = 0
        textgröße = 12
        symbol = TextIntern()
        symbol.PositionSetzen(x: x, y: y)
        symbol.FarbeSetzen(farbe: farbe)
        symbol.SichtbarkeitSetzen(sichtbar: sichtbar)
        symbol.WinkelSetzen(winkel: winkel)
        symbol.TextGrößeSetzen(größe: textgröße)
    }
    
    /**
    Setzt die Position (der Grundline) des Dreiecks.
    - parameters:
        - x: x-Position der linken Seite der Grundlinie
        - y: y-Position der Grundlinie
    */
    open func PositionSetzen (x: Int, y: Int)
    {
        self.x = x
        self.y = y
        symbol.PositionSetzen(x: x, y: y)
    }
    
    /**
    Setzt den aktuellen Text.
    - parameters:
        - text: der neue Text
    */
    open func TextSetzen (text: String)
    {
        symbol.TextSetzen(t: text)
    }
    
    /**
    Setzt die Größe des Textes.
    - parameters:
        - größe: die (neue) Textgröße
    */
    open func TextGrößeSetzen (größe: Int)
    {
        textgröße = größe
        symbol.TextGrößeSetzen(größe: größe)
    }
    
    /**
    Vergrößert den Text.
    */
    open func TextVergrößern()
    {
        symbol.TextVergrößern()
        textgröße = Int(symbol.size)
    }
    
    /**
    Verkleinert den Text.
    */
    open func TextVerkleinern()
    {
        symbol.TextVerkleinern()
        textgröße = Int(symbol.size)
    }
    
    /**
    Verschiebt den Text um die angegebenen Werte.
    - parameters:
        - deltaX: Verschiebung in x-Richtung
        - deltaY: Verschiebung in y-Richtung
    */
    open func Verschieben (deltaX: Int, deltaY: Int)
    {
        x += deltaX
        y += deltaY
        symbol.PositionSetzen(x: x, y: y)
    }

    /**
    Dreht das Dreieck
    - parameters:
        - grad: Drehwinkel (mathematisch positiver Drehsinn) im Gradmass
    */
    open func Drehen (grad: Int)
    {
        winkel += grad
        symbol.WinkelSetzen(winkel: winkel)
    }
    
    /**
    Setzt die Farbe des Textes.
    Erlaubte Farben sind:
    "weiß", "weiss", "rot", "grün", "gruen", "blau", "gelb",
    "magenta", "cyan", "hellgelb", "hellgrün", "hellgruen",
    "orange", "braun", "grau", "schwarz"
    Alle anderen Eingaben werden auf die Farbe schwarz abgebildet.
    - parameters:
        - farbe: (neue) Farbe
    */
    open func FarbeSetzen (farbe: String)
    {
        self.farbe = farbe
        symbol.FarbeSetzen(farbe: farbe)
    }
    
    /**
    Setzt den Drehwinkel des Textes.
    Die Winkelangabe ist in Grad,positive Werte drehen gegen den Uhrzeigersinn,
    negative Werte drehen im Uhrzeigersinn (mathematisch positiver Drehsinn).
    - parameters:
        - winkel: der (neue) Drehwinkel des Textes
    */
    open func WinkelSetzen (winkel: Int)
    {
        self.winkel = winkel
        symbol.WinkelSetzen(winkel: winkel)
    }
    
    /**
    Schaltet die Sichtbarkeit des Textes ein oder aus.
    Erlaubte Parameterwerte: true, false
    - parameters:
        - sichtbar: (neue) Sichtbarkeit des Textes
    */
    open func SichtbarkeitSetzen (sichtbar: Bool)
    {
        self.sichtbar = sichtbar
        symbol.SichtbarkeitSetzen(sichtbar: sichtbar)
    }
    
    /**
    Entfernt den Text aus dem Zeichenfenster.
    */
    open func Entfernen ()
    {
        symbol.Entfernen()
    }

    /**
    Bringt den Text eine Ebene nach vorn.
    */
    open func NachVornBringen()
    {
        symbol.NachVornBringen()
    }
    
    /**
    Bringt den Text in die vorderste Ebene.
    */
    open func GanzNachVornBringen()
    {
        symbol.GanzNachVornBringen()
    }
    
    /**
    Bringt den Text eine Ebene nach hinten.
    */
    open func NachHintenBringen()
    {
        symbol.NachHintenBringen()
    }
    
    /**
    Bringt den Text in die hinterste Ebene.
    */
    open func GanzNachHintenBringen()
    {
        symbol.GanzNachHintenBringen()
    }
}

/**
Wrapperklasse für die Turtle auf der Zeichenfläche.
*/
open class Turtle: AktionsEmpfänger
{
    /** x-Position der Turtle. */
    public var x: Int
    /** y-Position der Turtle. */
    public var y: Int
    /** Größe der Turtle. */
    public var größe: Int
    /** Farbe der Turtle. */
    public var farbe: String
    /** Sichtbarkeit der Turtle. */
    public var sichtbar: Bool
    /** Drehwinkel (mathematisch positiver Drehsinn) der Turtle in Grad. */
    public var winkel: Int
    /** Stiftposition. */
    public var stiftUnten: Bool
    /** Referenz auf das Delegate-Objekt. */
    var symbol: TurtleIntern
    
    /**
    Der Konstruktor erzeugt das Delegate-Objekt
    */
    public init()
    {
        x = 10
        y = 10
        größe = 40
        farbe = "schwarz"
        sichtbar = true
        winkel = 0
        stiftUnten = true
        symbol = TurtleIntern()
        symbol.SichtbarkeitSetzen(sichtbar: true)
        x = Int(symbol.x)
        y = Int(symbol.y)
        winkel = symbol.winkel
        größe = Int(symbol.breite)
        stiftUnten = symbol.stiftUnten
        sichtbar = symbol.sichtbar
        MeineView.v.AktionsEmpfängerEintragen(neu: self)
    }
    
    /**
    Methode wird aufgerufen, wenn die Turtle handeln soll.
    Die vordefinierte Methode tut nichts.
    */
    open func AktionAusführen()
    {
    }
    
    /**
    Die eigentliche Aktionsmethode für gedrückte Tasten.
    
    Muss bei Bedarf von einer Unterklasse überschrieben werden.
    - parameters:
    	- taste: die gedrückte Taste
    */
    open func TasteGedrückt (taste: Character)
    {
        //print("Taste: \(taste)")
    }
    
    /**
    Die eigentliche Aktionsmethode für gedrückte Sondertasten.
    
    Muss bei Bedarf von einer Unterklasse überschrieben werden.
    - parameters:
        - taste: KeyCode der gedrückten Taste
     * @param taste KeyCode der gedrückten Taste
    */
    open func SonderTasteGedrückt (taste: Int)
    {
        //print("Sondertaste: \(taste)")
    }
    
    /**
    Die eigentliche Aktionsmethode für einen Mausklick.
    
    Muss bei Bedarf von einer Unterklasse überschrieben werden.
    - parameters:
        -x: x-Position des Mausklicks
        -y: y-Position des Mausklicks
        - anzahl: Anzahl der aufeinanderfolgenden Mausklicks
    */
    open func MausGeklickt(x:Int, y:Int, anzahl:Int)
    {
        print("Maus: (\(x)|\(y)), \(anzahl) mal")
    }

    /**
    Setzt die Position der Turtle.
    - parameters:
        - x: x-Position der Turtle
        - y: y-Position der Turtle
    */
    
    open func PositionSetzen (x: Int, y: Int)
    {
        self.x = x
        self.y = y
        symbol.PositionSetzen(x: x, y: y)
    }
    
    /**
    Setzt die Größe des Turtlesymbols.
    - parameters:
        - größe: (neue) Größe
    */
    open func GrößeSetzen (größe: Int)
    {
        self.größe = größe
        symbol.GrößeSetzen(breite: größe, höhe: größe)
    }
    
    /**
    Setzt die Farbe der Linie.
    Erlaubte Farben sind:
    "weiß", "weiss", "rot", "grün", "gruen", "blau", "gelb",
    "magenta", "cyan", "hellgelb", "hellgrün", "hellgruen",
    "orange", "braun", "grau", "schwarz"
    Alle anderen Eingaben werden auf die Farbe schwarz abgebildet.
    - parameters:
        - farbe: (neue) Farbe
    */
    open func FarbeSetzen (farbe: String)
    {
        self.farbe = farbe
        symbol.FarbeSetzen(farbe: farbe)
    }

    /**
    Setzt den Drehwinkel der Turtle.
    Die Winkelangabe ist in Grad,positive Werte drehen gegen den Uhrzeigersinn,
    negative Werte drehen im Uhrzeigersinn (mathematisch positiver Drehsinn).
    0˚: Turtle schaut nach rechts,
    90˚: Turtle schaut nach oben,
    180˚: Turtle schaut nach links,
    270˚bzw. -90˚: Turtle schaut nach unten
    - parameters:
        - winkel: der (neue) Drehwinkel der Turtle
    */
    open func WinkelSetzen (winkel: Int)
    {
        self.winkel = winkel
        symbol.WinkelSetzen(winkel: winkel)
    }

    /**
    Schaltet die Sichtbarkeit der Turtle ein oder aus.
    Erlaubte Parameterwerte: true, false
    - parameters:
        - sichtbar: (neue) Sichtbarkeit der Turtle
    */
    open func SichtbarkeitSetzen (sichtbar: Bool)
    {
        self.sichtbar = sichtbar
        symbol.SichtbarkeitSetzen(sichtbar: sichtbar)
    }
    
    /**
    Entfernt die Turtle aus dem Zeichenfenster.
    */
    open func Entfernen ()
    {
    	MeineView.v.AktionsEmpfängerEntfernen(alt: self)
        symbol.Entfernen()
    }

    /**
    Bringt die Turtle eine Ebene nach vorn.
    */
    open func NachVornBringen()
    {
        symbol.NachVornBringen()
    }
    
    /**
    Bringt die Turtle in die vorderste Ebene.
    */
    open func GanzNachVornBringen()
    {
        symbol.GanzNachVornBringen()
    }
    
    /**
    Bringt die Turtle eine Ebene nach hinten.
    */
    open func NachHintenBringen()
    {
        symbol.NachHintenBringen()
    }
    
    /**
    Bringt die Turtle in die hinterste Ebene.
    */
    open func GanzNachHintenBringen()
    {
        symbol.GanzNachHintenBringen()
    }

    /**
    Setzt die Turtle wieder an ihre Ausgangsposition.
    */
    open func ZumStartpunktGehen()
    {
        symbol.ZumStartpunktGehen()
        x = Int(symbol.x)
        y = Int(symbol.y)
        winkel = symbol.winkel
    }

    /**
    Bewegt die Turtle nach vorne.
    - parameters:
        - länge: Anzahl der Längeneinheiten
    */
    open func Gehen(länge: Double)
    {
        symbol.Gehen(länge: CGFloat(länge))
        x = Int(symbol.x);
        y = Int(symbol.y);
    }

    /**
    Dreht die Turtle
    - parameters:
        - grad: Drehwinkel (mathematisch positiver Drehsinn) im Gradmass
    */
    open func Drehen (grad: Int)
    {
        symbol.Drehen(grad: grad)
        winkel = symbol.winkel
    }

    /**
    Versetzt Zeichenfläche und Turtle in den Ausgangszustand
    */
    open func Löschen()
    {
        symbol.Löschen()
    }

    /**
    Turtle wechselt in den Modus "nicht zeichnen"
    */
    open func StiftHeben()
    {
        symbol.StiftHeben()
        stiftUnten = false
    }

    /**
    Turtle wechselt in den Modus "zeichnen"
    */
    open func StiftSenken()
    {
        symbol.StiftSenken()
        stiftUnten = true
    }

    /**
    Gibt den aktuellen Winkel der Turtle zurück.
    Die Winkelangabe ist in Grad, positive Werte drehen gegen den Uhrzeigersinn,
    negative Werte drehen im Uhrzeigersinn (mathematisch positiver Drehsinn), d. h.
    0˚: Turtle schaut nach rechts,
    90˚: Turtle schaut nach oben,
    180˚: Turtle schaut nach links,
    270˚bzw. -90˚: Turtle schaut nach unten
    - returns: Winkel im Gradmass
    */
    open func WinkelGeben() -> Int
    {
        return winkel
    }

    /**
    Gibt die x-Koordinate der Turtle zurück
    - returns: x-Koordinate
    */
    open func XPositionGeben() -> Int
    {
        return x
    }
 
    /**
    Gibt die y-Koordinate der Turtle zurück
    - returns: y-Koordinate
    */
    open func YPositionGeben() -> Int
    {
        return y
    }
 
    /**
    Schaltet die Sichtbarkeit des Turtlesymbols ein oder aus.
    Erlaubte Parameterwerte: true, false
    - parameters:
        - sichtbar: (neue) Sichtbarkeit des Turtlesymbols
    */
    open func SichtbarkeitFürSymbolSetzen (sichtbar: Bool)
    {
        symbol.SichtbarkeitFürSymbolSetzen(sichtbar: sichtbar)
    }
 
    /**
    Testet, ob die Turtle eine Figur  berührt.
    - returns: true, wenn die Turtlekoordinaten innerhalb einer Grafikfigur sind
    */
    open func Berührt () -> Bool
    {
        return symbol.Berührt()
    }
 
    /**
    Testet, ob die Turtle eine Figur in der angegebenen Farbe berührt.
    - parameters:
        - farbe: die Farbe, die die berührte Figur haben muss.
    - returns: true, wenn die Turtlekoordinaten innerhalb einer Grafikfigur in der angegebenen Farbe sind
    */
    open func Berührt (farbe: String) -> Bool
    {
        return symbol.Berührt(farbe: farbe)
    }

    /**
    Testet, ob die Turtle eine Figur in der angegebenen Farbe berührt.
    - parameters:
        - objekt: das Objekt, das getestet werden soll.
    - returns: true, wenn die Turtlekoordinaten innerhalb einer Grafikfigur in der angegebenen Farbe sind
    */
    open func Berührt (objekt: AnyObject) -> Bool
    {
        return symbol.Berührt(object: objekt)
    }
}

/**
Wrapperklasse für die Figur auf der Zeichenfläche.
*/
open class Figur: AktionsEmpfänger
{
    /** x-Position der Figur. */
    private var x: Int
    /** y-Position der Figur. */
    private var y: Int
    /** Größe der Figur. */
    private var größe: Int
    /** Sichtbarkeit der Figur. */
    private var sichtbar: Bool
    /** Drehwinkel (mathematisch positiver Drehsinn) der Figur in Grad. */
    private var winkel: Int
    /** Referenz auf das Delegate-Objekt. */
    var symbol: FigurIntern
    
    /**
    Der Konstruktor erzeugt das Delegate-Objekt
    */
    public init()
    {
        x = 10
        y = 10
        größe = 40
        sichtbar = true
        winkel = 0
        symbol = FigurIntern()
        symbol.SichtbarkeitSetzen(sichtbar: true)
        x = Int(symbol.x)
        y = Int(symbol.y)
        winkel = symbol.winkel
        größe = Int(symbol.breite)
        sichtbar = symbol.sichtbar
        MeineView.v.AktionsEmpfängerEintragen(neu: self)
    }
    
    /**
    Methode wird aufgerufen, wenn die Turtle handeln soll.
    Die vordefinierte Methode tut nichts.
    */
    open func AktionAusführen()
    {
    }
    
    /**
    Die eigentliche Aktionsmethode für gedrückte Tasten.
    
    Muss bei Bedarf von einer Unterklasse überschrieben werden.
    - parameters:
        - taste: die gedrückte Taste
    */
    open func TasteGedrückt (taste: Character)
    {
        //print("Taste: \(taste)")
    }
    
    /**
    Die eigentliche Aktionsmethode für gedrückte Sondertasten.
    
    Muss bei Bedarf von einer Unterklasse überschrieben werden.
    - parameters:
        - taste: KeyCode der gedrückten Taste
     * @param taste KeyCode der gedrückten Taste
    */
    open func SonderTasteGedrückt (taste: Int)
    {
        //print("Sondertaste: \(taste)")
    }
    
    /**
    Die eigentliche Aktionsmethode für einen Mausklick.
    
    Muss bei Bedarf von einer Unterklasse überschrieben werden.
    - parameters:
        -x: x-Position des Mausklicks
        -y: y-Position des Mausklicks
        - anzahl: Anzahl der aufeinanderfolgenden Mausklicks
    */
    open func MausGeklickt(x:Int, y:Int, anzahl:Int)
    {
        print("Maus: (\(x)|\(y)), \(anzahl) mal")
    }

    /**
    Setzt die Position der Figur.
    - parameters:
        - x: x-Position der Figur
        - y: y-Position der Figur
    */
    open func PositionSetzen (x: Int, y: Int)
    {
        self.x = x
        self.y = y
        symbol.PositionSetzen(x: x, y: y)
    }
    
    /**
    Setzt die Größe des Figursymbols.
    - parameters:
        - größe: (neue) Größe
    */
    open func GrößeSetzen (größe: Int)
    {
        self.größe = größe
        symbol.GrößeSetzen(breite: größe, höhe: größe)
    }

    /**
    Setzt den Drehwinkel der Figur.
    Die Winkelangabe ist in Grad,positive Werte drehen gegen den Uhrzeigersinn,
    negative Werte drehen im Uhrzeigersinn (mathematisch positiver Drehsinn).
    0˚: Figur schaut nach rechts,
    90˚: Figur schaut nach oben,
    180˚: Figur schaut nach links,
    270˚bzw. -90˚: Figur schaut nach unten
    - parameters:
        - winkel: der (neue) Drehwinkel der Figur
    */
    open func WinkelSetzen (winkel: Int)
    {
        self.winkel = winkel
        symbol.WinkelSetzen(winkel: winkel)
    }

    /**
    Schaltet die Sichtbarkeit der Figur ein oder aus.
    Erlaubte Parameterwerte: true, false
    - parameters:
        - sichtbar: (neue) Sichtbarkeit der Figur
    */
    open func SichtbarkeitSetzen (sichtbar: Bool)
    {
        self.sichtbar = sichtbar
        symbol.SichtbarkeitSetzen(sichtbar: sichtbar)
    }
    
    /**
    Entfernt die Figur aus dem Zeichenfenster.
    */
    open func Entfernen ()
    {
        MeineView.v.AktionsEmpfängerEntfernen(alt: self)
        symbol.Entfernen()
    }

    /**
    Bringt die Figur eine Ebene nach vorn.
    */
    open func NachVornBringen()
    {
        symbol.NachVornBringen()
    }
    
    /**
    Bringt die Figur in die vorderste Ebene.
    */
    open func GanzNachVornBringen()
    {
        symbol.GanzNachVornBringen()
    }
    
    /**
    Bringt die Figur eine Ebene nach hinten.
    */
    open func NachHintenBringen()
    {
        symbol.NachHintenBringen()
    }
    
    /**
    Bringt die Figur in die hinterste Ebene.
    */
    open func GanzNachHintenBringen()
    {
        symbol.GanzNachHintenBringen()
    }

    /**
    Setzt die Figur wieder an ihre Ausgangsposition.
    */
    open func ZumStartpunktGehen()
    {
        symbol.ZumStartpunktGehen()
        x = Int(symbol.x)
        y = Int(symbol.y)
        winkel = symbol.winkel
    }

    /**
    Bewegt die Figur nach vorne.
    - parameters:
        - länge: Anzahl der Längeneinheiten
    */
    open func Gehen(länge: Double)
    {
        symbol.Gehen(länge: CGFloat(länge))
        x = Int(symbol.x);
        y = Int(symbol.y);
    }

    /**
    Dreht die Figur
    - parameters:
        - grad: Drehwinkel (mathematisch positiver Drehsinn) im Gradmass
    */
    open func Drehen (grad: Int)
    {
        symbol.Drehen(grad: grad)
        winkel = symbol.winkel
    }

    /**
    Gibt den aktuellen Winkel der Figur zurück
    0˚: Figur schaut nach rechts,
    90˚: Figur schaut nach oben,
    180˚: Figur schaut nach links,
    270˚bzw. -90˚: Figur schaut nach unten
    - returns: Winkel im Gradmass
    */
    open func WinkelGeben() -> Int
    {
        return winkel
    }

    /**
    Gibt die x-Koordinate der Figur zurück
    - returns: x-Koordinate
    */
    open func XPositionGeben() -> Int
    {
        return x
    }
 
    /**
    Gibt die y-Koordinate der Figur zurück
    - returns: y-Koordinate
    */
    open func YPositionGeben() -> Int
    {
        return y
    }
 
    /**
    Testet, ob die Figur eine Grafik-Figur berührt.
    - returns: true, wenn die Figur und eine Grafikfigur überlappen
    */
    open func Berührt () -> Bool
    {
        return symbol.Berührt()
    }
 
    /**
    Testet, ob die Figur eine Grafik-Figur in der angegebenen Farbe berührt.
    - parameters:
        - farbe: die Farbe, die die berührte Figur haben muss.
    - returns: true, wenn die Figur und eine Grafikfigur in der angegebenen Farbe überlappen
    */
    open func Berührt (farbe: String) -> Bool
    {
        return symbol.Berührt(farbe: farbe)
    }

    /**
    Testet, ob die Figur die angegebene Figur berührt.
    - parameters:
        - objekt: das Objekt, das getestet werden soll.
    - returns: true, wenn die Figur die angegebene Grafikfigur überlappen
    */
    open func Berührt (objekt: AnyObject) -> Bool
    {
        return symbol.Berührt(object: objekt)
    }
    
    /**
    Erzeugt ein neues, rechteckiges Element einer eigenen Darstellung der Figur.
 
    Alle Werte beziehen sich auf eine Figur der Größe 100x100 und den Koordinaten (0|0) in der Mitte des Quadrats
    - parameters:
        - x: x-Wert der linken oberen Ecke des Rechtecks
        - y: y-Wert der linken oberen Ecke des Rechtecks
        - breite: Breite des Rechtecks
        - höhe: Höhe des Rechtecks
        - farbe: (Füll)Farbe des Rechtecks
    */
    open func FigurteilFestlegenRechteck (x: Int, y: Int, breite: Int, höhe: Int, farbe: String)
    {
        symbol.FigurteilFestlegenRechteck(x: x, y: y, breite: breite, höhe: höhe, farbe: farbe)
    }
    
    /**
    Erzeugt ein neues, elliptisches Element einer eigenen Darstellung der Figur.
 
    Alle Werte beziehen sich auf eine Figur der Größe 100x100 und den Koordinaten (0|0) in der Mitte des Quadrats
    - parameters:
        - x: x-Wert der linken oberen Ecke des umgebenden Rechtecks der Ellipse
        - y: y-Wert der linken oberen Ecke des umgebenden Rechtecks der Ellipse
        - breite: Breite des umgebenden Rechtecks der Ellipse
        - höhe: Höhe des umgebenden Rechtecks der Ellipse
        - farbe: (Füll)Farbe der Ellipse
    */
    open func FigurteilFestlegenEllipse (x: Int, y: Int, breite: Int, höhe: Int, farbe: String)
    {
        symbol.FigurteilFestlegenEllipse(x: x, y: y, breite: breite, höhe: höhe, farbe: farbe)
    }

    /**
    Erzeugt ein neues, dreieckiges Element einer eigenen Darstellung der Figur.
     Alle Werte beziehen sich auf eine Figur der Größe 100x100 und den Koordinaten (0|0) in der Mitte des Quadrats
     - parameters:
         - x1: x-Wert des ersten Punkts des Dreiecks
         - y1: y-Wert des ersten Punkts des Dreiecks
         - x2: x-Wert des zweiten Punkts des Dreiecks
         - y2: y-Wert des zweiten Punkts des Dreiecks
         - x3: x-Wert des dritten Punkts des Dreiecks
         - y3: y-Wert des dritten Punkts des Dreiecks
         - farbe (Füll)Farbe des Dreiecks
    */
    open func FigurteilFestlegenDreieck(x1: Int, y1: Int, x2: Int, y2: Int, x3: Int, y3: Int, farbe: String)
    {
        symbol.FigurteilFestlegenDreieck(x1: x1, y1: y1, x2: x2, y2: y2, x3: x3, y3: y3, farbe: farbe)
    }

    /**
    Löscht die Vereinbarung für die eigene Figur.
    
    Die Figur wird wieder durch die Originalfigur dargestellt.
    */
    open func EigeneFigurLöschen()
    {
        symbol.EigeneFigurLöschen()
    }
}

/**
Zugriff auf die Ereignisse einschließlich Taktgeber.
*/
open class Ereignisbehandlung: AktionsEmpfänger
{
	public init()
	{
        if MeineView.v == nil
        {
            PlaygroundPage.current.liveView = GesamtFenster(frame: CGRect (x: 0.0, y: 0.0, width: anzeigebreite, height: anzeigeHöhe + zusatzHöhe))
        }
        MeineView.v.AktionsEmpfängerEintragen(neu: self)
    }
    
    /**
    Die eigentliche Aktionsmethode des Zeitgebers.
    Die vordefinierte Methode ruft TaktImpulsAusführen auf.
    */
    func AktionAusführen()
    {
    	TaktImpulsAusführen()
    }

    /**
    Die eigentliche Aktionsmethode des Zeitgebers.
    
    Muss bei Bedarf von einer Unterklasse überschrieben werden.
    */
    open func TaktImpulsAusführen()
    {
    	print("Tick")
    }
    /**
    Zeitgeber starten.
    */
    open func Starten ()
    {
        #if os(macOS)
        MeineView.v.Start()
        #elseif os(iOS)
        MeineView.g.TimerStart(g: MeineView.g)
        #endif
    }

    /**
    Zeitgeber anhalten.
    */
    open func Anhalten ()
    {
        #if os(macOS)
        MeineView.v.Stop();
        #elseif os(iOS)
        MeineView.g.TimerStop(g: MeineView.g)
        #endif
    }
 
    /**
    Ablaufgeschwindigkeit des Zeitgebers einstellen.
    - parameters:
    	- dauer: Angabe in Millisekunden
     */
    open func TaktdauerSetzen (dauer: Int)
    {
        MeineView.v.TaktlängeSetzen(s: Double(dauer) / 1000.0)
        #if os(iOS)
        MeineView.g.SliderSetzen(wert: dauer)
        #endif
    }

    /**
    Die eigentliche Aktionsmethode für gedrückte Tasten.
    
    Muss bei Bedarf von einer Unterklasse überschrieben werden.
    - parameters:
        - taste: die gedrückte Taste
    */
    open func TasteGedrückt (taste: Character)
    {
        //print("Taste: \(taste)")
    }
    
    /**
    Die eigentliche Aktionsmethode für gedrückte Sondertasten.
    
    Muss bei Bedarf von einer Unterklasse überschrieben werden.
    - parameters:
        - taste: KeyCode der gedrückten Taste
     * @param taste KeyCode der gedrückten Taste
    */
    open func SonderTasteGedrückt (taste: Int)
    {
        //print("Sondertaste: \(taste)")
    }
    
    /**
    Die eigentliche Aktionsmethode für einen Mausklick.
    
    Muss bei Bedarf von einer Unterklasse überschrieben werden.
    - parameters:
        -x: x-Position des Mausklicks
        -y: y-Position des Mausklicks
        - anzahl: Anzahl der aufeinanderfolgenden Mausklicks
    */
    open func MausGeklickt(x:Int, y:Int, anzahl:Int)
    {
        print("Maus: (\(x)|\(y)), \(anzahl) mal")
    }

}
