import Foundation

/**
Kiste beim Erzeuger-Verbraucher-Problem

- author: Johannes Neumeyer
- version: 1.0
*/
class Kiste
{
    /** Laufende Nummer der Kiste */
    private var nummer: Int
    
    /**
    Legt die Kistennummer fest.
    - parameters:
        - nummerNeu: Nummer der Kiste
    */
    init(nummerNeu: Int)
    {
        nummer = nummerNeu
    }
    
    /**
    Meldet die Nummer der Kiste.
    - returns: Kistennummer
    */
    func KistennummerGeben() -> Int
    {
        return nummer
    }
}

/**
Zwischenspeicher beim Erzeuger-Verbraucher-Problem

- author: Johannes Neumeyer
- version 1.0
*/
class Speicher
{
    /** Gibt an, ob Platz frei ist. */
    private var frei: Bool
    /** eventuell vorhandene Kiste */
    private var kiste: Kiste?
    /** Das Monitorobjekt */
    private let monitor: NSCondition
    
    /**
    Besetzt die Attribute vor.
     */
    init()
    {
        frei = true
        kiste = nil
        monitor = NSCondition()
    }
    
    /**
    Ablegen der Kiste
    - parameters:
        - kiste: die abzulegende Kiste
    */
    func Ablegen(kisteNeu: Kiste)
    {
        monitor.lock()
        while !frei
        {
            monitor.wait()
        }
        frei = false
        kiste = kisteNeu
        monitor.signal()
        monitor.unlock()
    }
    
    /**
    Holen der Kiste
    - returns: die geholte Kiste
    */
    func Holen() -> Kiste
    {
        monitor.lock()
        while frei
        {
            monitor.wait()
        }
        frei = true
        let k = kiste
        kiste = nil
        monitor.signal()
        monitor.unlock()
        return k!
    }
}

/**
Erzeuger von Kisten.

- author: Johannes Neumeyer
- version: 1.0
*/
class Erzeuger: Thread
{
    /** Produktionszeit pro Kiste in Millisekunden */
    private var zeit: Int
    /** Zwischenspeicher */
    private var speicher: Speicher
    /** laufende Nummer der produzierten Kisten */
    private var nummer: Int

    /**
    Konstruktor für Objekte der Klasse Erzeuger
    - parameters:
        - zeitNeu: Produktionszeit für die Kisten
        - speicherNeu: der Speicher, in dem abgelegt werden soll
    */
    init(zeitNeu: Int, speicherNeu: Speicher)
    {
        zeit = zeitNeu
        speicher = speicherNeu
        nummer = 0
    }

    /**
    Die Arbeitsmethode des Threads.
    */
    override func main()
    {
        while true
        {
            let kNeu = Produzieren()
            print("Ablegeversuch von Kiste \(kNeu.KistennummerGeben())")
            speicher.Ablegen(kisteNeu: kNeu)
            print("Kiste \(kNeu.KistennummerGeben()) abgelegt.")
        }

    }

    /**
    Produziert Kisten in der gegebenen Zeit.
    - returns: die neue Kiste
    */
    func Produzieren() -> Kiste
    {
        Thread.sleep(forTimeInterval: Double(zeit) / 1000.0)
        
        nummer = nummer + 1
        print("Kiste Nummer \(nummer) produziert.")
        return Kiste(nummerNeu: nummer)
    }
}


/**
Verbraucher von Kisten.

- author :Johannes Neumeyer
- version: 1.0
*/
class Verbraucher: Thread
{
    /** Die Produktionszeit pro Kiste in Millisekunden */
    private var zeit: Int
    /** Der Zwischenspeicher */
    private var speicher: Speicher

    /**
    Konstruktor für Objekte der Klasse Verbraucher
    - parameters:
        - zeitNeu: Ablagezeit für die Kisten
        - speicherNeu: der Speicher, aus dem geholt werden soll
    */
    init(zeitNeu: Int, speicherNeu: Speicher)
    {
        zeit = zeitNeu
        speicher = speicherNeu
    }

    /**
    Die Arbeitsmethode des Threads
    */
    override func main()
    {
        while true
        {
            let kNeu = speicher.Holen()
            Einlagern(kisteNeu: kNeu)
        }
    }

    /**
    Lagert Kisten in der gegebenen Zeit ein.
    - parameters:
        - kisteNeu die zu lagernde Kiste
    */
    func Einlagern(kisteNeu: Kiste)
    {
        Thread.sleep(forTimeInterval: Double.random(in: 0.0 ... 0.001) * Double(zeit))
       print("Kiste Nummer \(kisteNeu.KistennummerGeben()) eingelagert.")
    }
}


/**
Erzeuger-Verbraucher-Problem

- author: Johannes Neumeyer
- version: 1.0
*/
class ErzeugerVerbraucherProblem
{

    /**
    Konstruktor für Objekte der Klasse ErzeugerVerbraucherProblem
    */
    init()
    {
        let speicher = Speicher()
        let erzeuger = Erzeuger(zeitNeu: 500, speicherNeu: speicher)
        let verbraucher = Verbraucher(zeitNeu: 2500, speicherNeu: speicher)
        erzeuger.start()
        verbraucher.start()
    }
}

ErzeugerVerbraucherProblem()
