import Foundation

/**
Gabel zwischen zwei Philosophen

- author: Johannes Neumeyer
- version: 1.0
*/
class Gabel: Ereignisbehandlung
{
    /** gibt an, ob die Gabel aktuell genutzt wird */
    private var inBenutzung: Bool
    /** Id der Gabel */
    private var id: Int
    /** Das Rechteck zur Darstellung */
    private var symbol: Rechteck
    /** Zu setzende Gabelfarbe */
    private var gabelFarbe: String
    /** Monitor */
    private var monitor: NSCondition

    /**
    Konstruktor für Objekte der Klasse Gabel
     - parameters:
        - gabelId: Id der Gabel
    */
    init(gabelId: Int, x: Int, y: Int, breite: Int, höhe: Int, grad: Int)
    {
        inBenutzung = false
        id = gabelId
        gabelFarbe = "schwarz"
        monitor = NSCondition()
        symbol = Rechteck()
        super.init()
        symbol.FarbeSetzen(farbe: "schwarz")
        symbol.PositionSetzen(x: x, y: y)
        symbol.GrößeSetzen(breite: breite, höhe: höhe)
        symbol.Drehen(grad: grad)
        Starten()
        TaktdauerSetzen(dauer: 10)
    }
    
    /**
     Setzt die Farbinformation, die im Hauptthread verarbeitet wird
     - parameters:
        - farbe: die zu setzende Farbe
     */
    private func FarbeSetzen(farbe: String)
    {
        gabelFarbe = farbe
    }
    
    /**
    Bei jedem Taktimpuls wird die gespeichertze Frabe in dei Anzeige übernommen.
    */
    override func TaktImpulsAusführen()
    {
        symbol.FarbeSetzen(farbe: gabelFarbe)
    }
    
    /**
    Es wird gewartet, bis die Gabel nicht mehr in Benutzung ist;
    dann wird sie aufgenommen.

    - parameters:
        - eigeneFarbe: Die Farbe des Philosophen, der die Gabel aufnehmen möchte; die Gabel wird dann auf diese Farbe gesetzt.
    */
    func Aufnehmen(eigeneFarbe: String)
    {
        monitor.lock()
        while inBenutzung
        {
            monitor.wait()
        }
        inBenutzung = true
        FarbeSetzen(farbe: eigeneFarbe)
        monitor.unlock()
    }
        
    /**
    Die Gabel wird abgelegt; da sie dann keinen Besitzer mehr hat, wird ihre Farbe auf "schwarz" gesetzt.
    */
    func Ablegen()
    {
        monitor.lock()
        inBenutzung = false
        FarbeSetzen(farbe: "schwarz")
        monitor.signal()
        monitor.unlock()
    }
    
    /**
    Liefert die Id der Gabel.
    - returns: Id der Gabel
    */
    func IdGeben() -> Int
    {
        return id
    }
}

/**
Speisender Philosoph

- author: Johannes Neumeyer
- version 1.0
*/
class Philosoph: Thread
{
    /** Id des Philosophen */
    private var id: Int
    /** Zeitangabe in ms als Grundlage für die Bestimmung zufälliger Ess- und Wartezeiten */
    private var wartezeit: Int
    /** Teller des Philosophen */
    private var teller: Kreis
    /** Farbe des Tellers */
    private var tellerfarbe: String
    /** Referenz auf die linke Gabel */
    private var gabelLinks: Gabel
    /** Referenz auf die rechte Gabel */
    private var gabelRechts: Gabel
    /** Zufallsgenerator */
    private var ran: Zufall

    /**
    Konstruktor für Objekte der Klasse Philosoph
    - parameters:
        - philosophenId :Id des Philosophen
        - tellerNeu: der Teller, von dem der Philosoph speist
        - tellerfarbeNeu: Farbe des Tellers
        - links: die linke Gabel, die der Philosoph nutzt
        - rechts: die rechte Gabel, die der Philosoph nutzt
     */
    init(philosophenId: Int, tellerNeu: Kreis, tellerfarbeNeu: String, links: Gabel, rechts: Gabel)
    {
        wartezeit = 500
        id = philosophenId
        teller = tellerNeu
        tellerfarbe = tellerfarbeNeu
        gabelLinks = links
        gabelRechts = rechts
        ran = Zufall()
    }

    /**
    Die Arbeitsmethode des Threads mit einer Endlosschleife:
    Der Philosoph nimmt nach einer Zeit des Denkens die Gabeln auf, isst
    und legt sie dann wieder ab.
    */
    override func main()
    {
        while true
        {
            Thread.sleep(forTimeInterval: Double(wartezeit) / 1000.0) // denken
            gabelLinks.Aufnehmen(eigeneFarbe: tellerfarbe)
            gabelRechts.Aufnehmen(eigeneFarbe: tellerfarbe)

            Thread.sleep(forTimeInterval: Double(wartezeit) / 1000.0) // essen
            gabelLinks.Ablegen()
            gabelRechts.Ablegen()
        }
    }
}

/**
Visualisierung des Problems der speisenden Philosophen

- author: Johannes Neumeyer
- version: 1.0
*/
class SpeisendePhilosophen
{
    /** verwaltet alle Gabeln */
    private var gabeln: [Gabel]
    /** verwaltet alle Teller */
    private var teller: [Kreis]
    /** verwaltet alle Tellerfarben */
    private var tellerfarben: [String]
    /** verwaltet alle Philosophen */
    private var philosophen: [Philosoph]
    /** instruierender Text oben im Zeichenfenster */
    private var anleitung: Text

    /**
    Beteiligte Objekte (Philosophen, Teller, Gabeln, ...) werden passend erstellt
    und die Philosophenthreads gestartet.
    */
    init()
    {
        gabeln = [Gabel]()
        teller = [Kreis]()
        tellerfarben = [String]()
        tellerfarben.append("rot")
        tellerfarben.append("blau")
        tellerfarben.append("grün")
        tellerfarben.append("magenta")
        tellerfarben.append("grau")
        philosophen = [Philosoph]()
        anleitung = Text();
        anleitung.TextSetzen(text: "Abgelegte Gabeln sind schwarz, aufgenommene Gabeln haben die Farbe ihres aktuellen Besitzers.")
        anleitung.PositionSetzen(x: 10, y: 50)
        anleitung.TextGrößeSetzen(größe: 17)

        for zähler in 0 ..< 5
        {
            gabeln.append(Gabel(gabelId: zähler, x: 350+Int(150*cos(Double(54+72*zähler)*Double.pi/180)), y: 295-Int (150*sin(Double(54+72*zähler)*Double.pi/180)), breite: 100, höhe: 10, grad: 54+72*zähler))

            teller.append(Kreis());
            teller[zähler].RadiusSetzen(radius: 50);
            teller[zähler].FarbeSetzen(farbe: tellerfarben[zähler]);
            teller[zähler].PositionSetzen(x: 400+Int(175*cos(Double(18+72*zähler)*Double.pi/180)), y: 300-Int (175*sin(Double(18+72*zähler)*Double.pi/180)))
        }
        
        for zähler in 0 ..< 5
        {
            philosophen.append(Philosoph(philosophenId: zähler, tellerNeu: teller[zähler], tellerfarbeNeu: tellerfarben[zähler], links: gabeln[(zähler - 1 + 5) % 5], rechts: gabeln[zähler]))
            philosophen[zähler].start()
        }
    }
}

SpeisendePhilosophen()

