import java.util.Random;
import java.util.ArrayList;
/**
 * Verwaltet einen ungerichteten, gewichteten Graphen mittels Adjazenzmatrix
 * 
 * @author Johannes Neumeyer, Albert Wiedemann
 * @version 1.0
 */
class Graph
{
    /** Feld der Knoten des Graphen */
    private ArrayList<Knoten> knoten;   
    /** Feld der Kanten des Graphen */
    private ArrayList<Kante> kanten;
    /** 2-dim Feld der Adjazenzmatrix */
    private ArrayList<ArrayList<Integer>> matrix;

    /**
     * Baut die Datenstruktur auf
     */
    Graph()
    {
        knoten = new ArrayList<Knoten>();
        kanten = new ArrayList<Kante>();
        matrix = new ArrayList<ArrayList<Integer>>();
    }

    /**
     * Einfügen eines neuen Knoten in den Graphen
     * @param bezeichner Bezeichner des neuen Knotens, der dem Graphen hinzugefügt wird.
     * @param x x-Koordinate für die Anzeige des Knotens
     * @param y y-Koordinate für die Anzeige des Knotens
     */
    void KnotenEinfügen(String bezeichner, int x, int y)
    {
        knoten.add(new Knoten(bezeichner, x, y));
        for (int index = 0; index < matrix.size(); index++)
        {
            matrix.get(index).add(new Integer(-1));
        }
        matrix.add(new ArrayList<Integer>());
        for (int index = 0; index < matrix.size(); index++)
        {
            matrix.get(matrix.size()-1).add(new Integer(-1));
        }
    }

    /**
     * Gibt den Knoten mit dem angegebenen Bezeichner zurück.
     * Wenn ein Knoten mit diesem Bezeichner nicht bekannt ist wird null zurückgegeben
     * @param bezeichner Bezeichner des Knoten der gesucht wird
     * @return Referenz auf das Knotenobjekt oder null
     */   
    private Knoten KnotenGeben(String bezeichner)
    {
        for (Knoten k: knoten)
        {
            if (k.BezeichnerGeben().equals(bezeichner))
            {
                return k;
            }
        }
        return null;
    }

    /**
     * Gibt die interne Nummer des Knotens
     * Wenn ein Knoten mit diesem Bezeichner nicht bekannt ist wird -1 zurückgegeben
     * @param bezeichner Bezeichner des Knoten der gesucht wird
     * @return Indexnummer des Knotens im Knotenfeld; 0 <= res <= knoten.size()-1 bzw. -1
     */   
    int KnotenNummerGeben(String bezeichner)
    {
        for (int index=0; index < knoten.size(); index++)
        {
            if (knoten.get(index).BezeichnerGeben().equals(bezeichner))
            {
                return index;
            }
        }
        return -1;
    }

    /**
     * Gibt die Bezeichnung eines Knotens mit der internen Knotennummer
     * @param Indexnummer des Knotens im Knotenarray; 0<= x <= knoten.size()
     * @return Bezeichner des Knotens
     */   
    String KnotenBezeichnerGeben(int knotenNummer)
    {
        if ((knotenNummer < knoten.size()) && (knotenNummer >= 0))
        {
            return knoten.get(knotenNummer).BezeichnerGeben();
        }
        else
        {
            return "";
        }
    }

    /**
     * Einfügen einer Kante in den Graphen
     * Eine Kante ist durch einen Anfangsknoten und einen Endknoten festgelegt und hat eine Gewichtung
     * @param von Bezeichner des Anfangsknotens
     * @param nach Bezeichner des Endknotens
     * @param gewichtung Gewichtung der Kante als Ganzzahl
     */
    void KanteEinfügen(String von, String nach, int gewichtung)
    {
        int vonNummer, nachNummer;
        vonNummer = KnotenNummerGeben(von);
        nachNummer = KnotenNummerGeben(nach);
        if ((vonNummer!=-1) && (nachNummer!=-1) && (vonNummer!=nachNummer))
        {
            matrix.get(vonNummer).set(nachNummer, gewichtung);
            matrix.get(nachNummer).set(vonNummer, gewichtung);
            Kante neueKante = new Kante(knoten.get(vonNummer), knoten.get(nachNummer), gewichtung);
            kanten.add(neueKante); // vorher könnte noch eine Prüfung eingebaut werden, ob die Kante schon vorhanden ist
        }
    }

    /**
     * Gibt die Adjazenzmatrix des Graphen in der Konsole aus
     * Nach Zeilen und Spalten formatiert
     * Als Spaltenbreite wurde hier 4 Zeichen gewählt.
     */   
    void Ausgeben()
    {
        // Kopfzeile
        System.out.print("    ");  
        for (int i = 0; i < knoten.size(); i++)
        {
            System.out.print((knoten.get(i).BezeichnerGeben() + "    ").substring(0, 4));
        }
        System.out.println();

        for (int i = 0; i < knoten.size(); i++)
        {
            System.out.print((knoten.get(i).BezeichnerGeben() + "    ").substring(0, 4));
            for (int j = 0; j < knoten.size(); j++)
            {
                if (matrix.get(i).get(j) != -1)
                {
                    System.out.print((matrix.get(i).get(j)+ "   ").substring(0, 4));
                }
                else
                {
                    System.out.print("    ");
                }
            }
            System.out.println();           
        }
    }    

    /**
     * Gibt die Anzahl der Knoten des Graphen zurück
     * @return  Anzahl der Knoten
     */   
    int KnotenAnzahlgeben()
    {
        return knoten.size();
    }

    /**
     * Gibt die Gewichtung einer Kante zurück
     * Die Kante ist durch einen Anfangsknoten und einen Endknoten festgelegt
     * @param von Bezeichner des Anfangsknotens
     * @param nach Bezeichner des Endknotens
     * @return Gewichtung der Kante
     */ 
    int KanteGewichtGeben(String von, String nach)
    {
        int vonNummer, nachNummer;

        vonNummer = KnotenNummerGeben(von);
        nachNummer = KnotenNummerGeben(nach);
        if ((vonNummer!=-1) && (nachNummer!=-1))
        {
            return matrix.get(vonNummer).get(nachNummer);
        }
        else
        {
            return -1;
        }
    }

    /**
     * Löscht die Kanten und Knoten des Graphen.
     */
    void ZurückSetzen()
    {

        for (Knoten k: knoten)
        {
            k.SymbolGeben().Entfernen();
        }
        for (Kante k : kanten)
        {
            k.KantensymbolGeben().Entfernen();
        }
        knoten.clear();
        kanten.clear();
        matrix.clear();
    }

    /**
     * Sucht rekursiv die Wurzel des Teilbaums, zu dem der gegebene Knoten gehört.
     * Wird im Rahmen des Union-Find-Algorithmus genutzt.
     * 
     * @param knoten Der Knoten, dessen Wurzel gesucht wird.
     * @return Die Wurzel des zugehörigen Teilbaums.
     */
    private Knoten Finden(Knoten knoten)
    {
        if (knoten.VorgängerGeben() == null)
        {
            return knoten;
        }
        else
        {
            return Finden(knoten.VorgängerGeben());
        }
    }

    /**
     * Vereinigt die Teilbäume, zu denen die beiden übergebenen Knoten gehören, indem der Wurzelknoten
     * des zweiten Baums auf den Wurzelknoten des ersten zeigt.
     * 
     * @param k1 ein Knoten aus dem ersten Teilbaum
     * @param k2 ein Knoten aus dem zweiten Teilbaum
     */
    private void Vereinigen(Knoten k1, Knoten k2)
    {
        Knoten wurzelk1 = Finden(k1);
        Knoten wurzelk2 = Finden(k2);
        wurzelk2.VorgängerSetzen(wurzelk1);
    }

    /**
     * Erzeugt eine Kopie der zentral verwalteten Kantenliste.
     * 
     * @return ArrayList aller im Graphen vorhandenen Kanten
     */
    private ArrayList<Kante> MatrixZuKantenliste()
    {        
        return new ArrayList<Kante>(kanten);
    }

    /**
     * Führt eine Preorder-Traversierung durch, beginnend bei einem zufällig ausgewählten Wurzelknoten
     * des minimalen Spannbaums, und gibt die besuchte Knotenfolge als Liste zurück.
     * 
     * @param minSpannbaum Kanten des minimalen Spannbaums
     * @return Liste der Knoten in Preorder-Reihenfolge
     */
    private ArrayList<Knoten> PreOrderGeben(ArrayList<Kante> minSpannbaum)
    {
        MinSpannbaumZuBaum(minSpannbaum);

        // Wurzel zufällig definieren
        Random zufallsgenerator = new Random();
        int zufallsindex = zufallsgenerator.nextInt(minSpannbaum.size());
        Knoten wurzel = minSpannbaum.get(zufallsindex).StartGeben();
        wurzel.FarbeSetzen("rot");            

        ArrayList<Knoten> preOrderFolge = new ArrayList<Knoten>();
        return PreOrderRekursiv(wurzel, preOrderFolge);     
    }

    /**
     * Rekursive Hilfsmethode für Preorder-Traversierungen auf Bäumen.
     * Fügt die Knoten in Preorder-Reihenfolge zur übergebenen Liste hinzu.
     * 
     * @param wurzel Der aktuelle Wurzelknoten
     * @param preOrderFolge bisher ermittelte Reihenfolge der Knoten
     * @return aktualisierte Preorder-Knotenfolge
     */
    private ArrayList<Knoten> PreOrderRekursiv(Knoten wurzel, ArrayList<Knoten> preOrderFolge)
    {
        Kante k;
        ArrayList<Knoten> preOrderFolgeLokal = new ArrayList<Knoten>(preOrderFolge);
        preOrderFolgeLokal.add(wurzel);
        wurzel.DfsBesuchtSetzen();
        ArrayList<Kante> baumnachbarn = wurzel.BaumnachbarnGeben();

        for (int i = 0; i < baumnachbarn.size(); i++)
        {
            k = baumnachbarn.get(i);            
            if (k.StartGeben().equals(wurzel) && !k.ZielGeben().DfsBesuchtGeben())
            {
                preOrderFolgeLokal = PreOrderRekursiv(k.ZielGeben(), preOrderFolgeLokal);
            }
            else if (k.ZielGeben().equals(wurzel) && !k.StartGeben().DfsBesuchtGeben())
            {
                preOrderFolgeLokal = PreOrderRekursiv(k.StartGeben(), preOrderFolgeLokal);
            }
        }
        return preOrderFolgeLokal;
    }

    /**
     * Initialisiert die Baum-Nachbarschaften der Knoten entsprechend dem minimalen Spannbaum.
     * Fügt die entsprechenden Kanten als Baumkanten zu den Knoten hinzu.
     * 
     * @param minSpannbaum Kanten des minimalen Spannbaums
     */
    private void MinSpannbaumZuBaum(ArrayList<Kante> minSpannbaum)
    {
        // Adjazenzen aller Knoten des minimalen Spannbaums setzen
        for (int i = 0; i < minSpannbaum.size(); i++)
        {
            minSpannbaum.get(i).StartGeben().BaumnachbarSetzen(minSpannbaum.get(i));
            minSpannbaum.get(i).ZielGeben().BaumnachbarSetzen(minSpannbaum.get(i));
        }
    }
}
