"""
Bestimmt die Laufzeit des Dijkstra-Algorithmus auf zufällig erzeugten Graphen.

@author Albert Wiedemann
@version 1.0
"""

import random
import time

class Knoten:
    """
    Knoteninformation fuer Dijkstra-Algorithmus.
    """

    def __init__(self, n):
        """
        Belegt die Attribute vor.
        :param n Nummer des Knotens
        """
        self.nummer = n
        self.laenge = 0

    def NummerGeben(self):
        """
        Meldet die Nummer des Knotens.
        :return Knotennummer
        """
        return self.nummer

    def LaengeSetzen(self, laengeNeu):
        """
        Setzt die Weglaenge.
        :param laengeNeu neue Weglaenge
        """
        self.laenge = laengeNeu

    def LaengeGeben(self):
        """
        Meldet die Weglaenge.
        :return die Weglaenge
        """
        return self.laenge


class Dijkstra:
    """
    Sucht im Graph nach dem Alorithmus von Dijkstra.
    """

    def __init__(self, m):
        """
        Legt die Hilfsdaten an.
        :param m Adjazenzmatrix 
        """
        self.matrix = m
        self.fertig = []
        self.rest = []
        self.inarbeit = []

    def WegSuchen(self, von, nach):
        """
        Fuehrt die Suche aus.
        :param von Startknoten
        :param nach Zielknoten
        :return Weglaenge
        """
        akt = von
        laenge = 0
        for i in range(len(self.matrix)):
            if i == von:
                self.fertig.append(Knoten(i))
            else:
                self.inarbeit.append(Knoten(i))
        while akt != nach:
            for i in range(len(self.matrix)):
                if self.matrix[akt][i] > 0:
                    k = self.KnotenSuchen(self.rest, i)
                    if k is not None:
                        k.LaengeSetzen(laenge + self.matrix[akt][i])
                        self.rest.remove(k)
                        self.inarbeit.append(k)
                    else:
                        k = self.KnotenSuchen(self.inarbeit, i)
                        if k is not None:
                            if laenge + self.matrix[akt][i] < k.LaengeGeben():
                                k.LaengeSetzen(laenge + self.matrix[akt][i])
            naechster = None
            for k in self.inarbeit:
                if (naechster is None) or (naechster.LaengeGeben() > k.LaengeGeben()):
                    naechster = k
            if naechster is None:
                return -1
            self.inarbeit.remove(naechster)
            self.fertig.append(naechster)
            laenge = naechster.LaengeGeben()
            akt = naechster.NummerGeben()
        return laenge

    def KnotenSuchen(self, feld, nummer):
        """
        Sucht den Knoten mit der angegebenen Nummer
        :param feld das Knotenfeld, in dem gesucht werden soll
        :param nummer die Nummer des gesuchten Knotens
        :return Knotenreferenz oder None, falls der Knoten mit der
        angegebenen Nummer nicht im Feld ist.
        """
        for k in feld:
            if k.NummerGeben() == nummer:
                return k
        return None


class Generieren:
    """
    Generiert den Graphen (Adjazenzmatrix).
    """

    maxEntfernung = 100

    def __init__(self, anzahl):
        """
        Legt die Felder an.
        :param anzahl Anzahl der Knoten
        """
        self.matrix = [[0 for _ in range(anzahl)] for _ in range(anzahl)]
        self.zufall = random.Random()

    def AdjazenzmatrixGenerieren(self, anzahl):
        """
        Generiert die Adjazenzmatrix.
        Auf eine zweidimensionale Darstellbarkeit wird nicht geachtet.
        :param anzahl Anzahl der zu generierenden Kanten
        :return Adjazenzmatrix
        """
        for i in range(len(self.matrix)):
            for j in range(len(self.matrix[i])):
                if i == j:
                    self.matrix[i][j] = 0
                else:
                    self.matrix[i][j] = -1
        for _ in range(anzahl):
            while True:
                von = self.zufall.randint(0, len(self.matrix) - 1)
                nach = self.zufall.randint(0, len(self.matrix[0]) - 1)
                if self.matrix[von][nach] < 0:
                    break
            weg = self.zufall.randint(0, self.maxEntfernung - 1)
            self.matrix[von][nach] = weg
            self.matrix[nach][von] = weg
        return self.matrix


class TestSuchen:
    """
    Fuehrt die gewuenschten Laufzeittests durch.
    """

    def __init__(self, anzahl):
        """
        Legt die benoetigten Objekte an und besetzt die Attributwerte.
        :param anzahl Anzahl der Knoten im Graph
        """
        self.anzahl = anzahl
        self.matrix = Generieren(anzahl).AdjazenzmatrixGenerieren((anzahl - 1) * 2)
        self.zeitDijkstra = 0

    def DurchschnittsTest(self):
        """
        Bestimmt die durchschnittliche Zeit fuer die Knotenbesuche.
        """
        zufall = random.Random()
        self.zeitDijkstra = 0
        for _ in range(1000):
            start = zufall.randint(0, self.anzahl - 1)
            while True:
                ziel = zufall.randint(0, self.anzahl - 1)
                if start != ziel:
                    break
            t = Dijkstra(self.matrix)
            startZeit = time.perf_counter_ns()
            t.WegSuchen(start, ziel)
            endeZeit = time.perf_counter_ns()
            self.zeitDijkstra += (endeZeit - startZeit)
        self.zeitDijkstra = self.zeitDijkstra // 1000

    def ZeitFuerDijkstraGeben(self):
        """
        Meldet die durchschnittliche Zeit fuer die Suche nach Dijkstra.
        :return Suchzeit
        """
        return self.zeitDijkstra


test = TestSuchen(10)  # z. B. 10 Knoten
test.DurchschnittsTest()
print("Durchschnittliche Zeit fuer Wegesuche nach Dijkstra:", test.ZeitFuerDijkstraGeben(), "us")