"""
Vergleicht die mittlere Suchzeit in sortierten Listen und in binaeren Suchbaeumen.

@author Albert Wiedemann
@version 1.0
"""

import random
import time

class BaumElement:
    """
    Beschreibt ein abstraktes Element des Baums.
    """
    def __init__(self):
        pass

    def Einfuegen(self, neu):
        """
        Fuegt ein neues Datenelement sortiert in den Baum ein.
        :param neu das neue Datenelement
        :return der (neue) Nachfolger fuer das Vorgaengerelement
        """
        raise NotImplementedError

    def Suchen(self, schluessel):
        """
        Sucht ein Datenelement mit dem angegebenen Schluessel.
        :param schluessel der zu suchende Schluesselwert
        :return das gefundene Datenelement oder None
        """
        raise NotImplementedError


class Abschluss(BaumElement):
    """
    Beschreibt ein Abschlusselement des Baums.
    """

    def __init__(self):
        """
        Fuer das Anlegen des Abschlusses brauchts nichts getan zu werden.
        """
        super().__init__()

    def Einfuegen(self, neu):
        """
        Fuegt ein neues Datenelement sortiert in den Baum ein.
        :param neu das neue Datenelement
        :return der (neue) Nachfolger fuer das Vorgaengerelement
        """
        return Knoten(neu, self, self)

    def Suchen(self, schluessel):
        """
        Sucht ein Datenelement mit dem angegebenen Schluessel.
        :param schluessel der zu suchende Schluesselwert
        :return das gefundene Datenelement oder None
        """
        return None


class Knoten(BaumElement):
    """
    Beschreibt einen Knoten des Baums.
    """

    def __init__(self, d, lnf, rnf):
        """
        Legt einen neuen Knoten mit gegebenem Datenelement und gegebenen
        Nachfolgern an.
        :param d das von diesem Knoten verwaltete Datenelement
        :param lnf Referenz auf das linke Nachfolgerelement
        :param rnf Referenz auf das rechte Nachfolgerelement
        """
        self.daten = d
        self.linksnachfolger = lnf
        self.rechtsnachfolger = rnf

    def Einfuegen(self, neu):
        """
        Fuegt ein neues Datenelement sortiert in den Baum ein.
        :param neu das neue Datenelement
        :return der (neue) Nachfolger fuer das Vorgaengerelement
        """
        if neu > self.daten:
            self.rechtsnachfolger = self.rechtsnachfolger.Einfuegen(neu)
        else:
            self.linksnachfolger = self.linksnachfolger.Einfuegen(neu)
        return self

    def Suchen(self, schluessel):
        """
        Sucht ein Datenelement mit dem angegebenen Schluessel.
        :param schluessel der zu suchende Schluesselwert
        :return das gefundene Datenelement oder None
        """
        vergleich = (schluessel > self.daten) - (schluessel < self.daten)
        if vergleich == 0:
            return self.daten
        elif vergleich < 0:
            return self.linksnachfolger.Suchen(schluessel)
        else:
            return self.rechtsnachfolger.Suchen(schluessel)


class Baum:
    """
    Ein sortierter Binaerbaum nach Standardimplementierung.
    """

    def __init__(self):
        """
        Legt einen leeren Baum an.
        """
        self.anfang = Abschluss()

    def Einfuegen(self, neu):
        """
        Fuegt ein neues Datenelement in den Baum ein.
        :param neu das neue Datenelement
        """
        self.anfang = self.anfang.Einfuegen(neu)

    def Suchen(self, schluessel):
        """
        Sucht ein Datenelement mit dem angegebenen Schluessel.
        :param schluessel der zu suchende Schluesselwert
        :return das gefundene Datenelement oder None
        """
        return self.anfang.Suchen(schluessel)


class ListenElement:
    """
    Beschreibt ein abstraktes Element der Liste.
    """
    def __init__(self):
        """
        Keine Aufgabe, nur zur Dokumentation aufgefuehrt.
        """
        pass

    def Einfuegen(self, neu):
        """
        Fuegt ein neues Datenelement sortiert in die Liste ein.
        :param neu das neue Datenelement
        :return der (neue) Nachfolger fuer das Vorgaengerelement
        """
        raise NotImplementedError

    def Suchen(self, schluessel):
        """
        Sucht ein Datenelement mit dem angegebenen Schluessel.
        :param schluessel der zu suchende Schluesselwert
        :return das gefundene Datenelement oder None
        """
        raise NotImplementedError


class ListenKnoten(ListenElement):
    """
    Beschreibt einen Knoten der Liste.
    """

    def __init__(self, d, nf):
        """
        Legt einen neuen Knoten mit gegebenem Datenelement und gegebenem
        Nachfolger an.
        :param d das von diesem Knoten verwaltete Datenelement
        :param nf Referenz auf das Nachfolgerelement
        """
        self.daten = d
        self.nachfolger = nf

    def Einfuegen(self, neu):
        """
        Fuegt ein neues Datenelement sortiert in die Liste ein.
        :param neu das neue Datenelement
        :return der (neue) Nachfolger fuer das Vorgaengerelement
        """
        if neu > self.daten:
            self.nachfolger = self.nachfolger.Einfuegen(neu)
            return self
        else:
            return ListenKnoten(neu, self)

    def Suchen(self, schluessel):
        """
        Sucht ein Datenelement mit dem angegebenen Schluessel.
        :param schluessel der zu suchende Schluesselwert
        :return das gefundene Datenelement oder None
        """
        if schluessel == self.daten:
            return self.daten
        else:
            return self.nachfolger.Suchen(schluessel)


class ListenAbschluss(ListenElement):
    """
    Beschreibt ein Abschlusselement der Liste.
    """

    def __init__(self):
        """
        Fuer das Anlegen des Abschlusses brauchts nichts getan zu werden.
        """
        super().__init__()

    def Einfuegen(self, neu):
        """
        Fuegt ein neues Datenelement sortiert in die Liste ein.
        :param neu das neue Datenelement
        :return der (neue) Nachfolger fuer das Vorgaengerelement
        """
        return ListenKnoten(neu, self)

    def Suchen(self, schluessel):
        """
        Sucht ein Datenelement mit dem angegebenen Schluessel.
        :param schluessel der zu suchende Schluesselwert
        :return das gefundene Datenelement oder None
        """
        return None


class Liste:
    """
    Eine sortierte Liste nach Standardimplementierung.
    """

    def __init__(self):
        """
        Legt eine leere Liste an.
        """
        self.anfang = ListenAbschluss()

    def Einfuegen(self, neu):
        """
        Fuegt ein neues Datenelement sortiert in die Liste ein.
        :param neu das neue Datenelement
        """
        self.anfang = self.anfang.Einfuegen(neu)

    def Suchen(self, schluessel):
        """
        Sucht ein Datenelement mit dem angegebenen Schluessel.
        :param schluessel der zu suchende Schluesselwert
        :return das gefundene Datenelement oder None
        """
        return self.anfang.Suchen(schluessel)


class Generator:
    """
    Generiert zufaellige Bezeichner vom Typ String gegebener Laenge.
    """

    def __init__(self, laenge):
        """
        Initialisiert den Zufallsgenerator.
        :param laenge Die Laenge der zu generierenden Bezeichner
        """
        self.zufall = random.Random()
        self.zeichen = ['a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j']
        self.laenge = laenge

    def BezeichnerGenerieren(self):
        """
        erzeugt einen neuen Bezeichner.
        :return der neue Bezeichner
        """
        resultat = ""
        for i in range(self.laenge):
            resultat = resultat + self.zeichen[self.zufall.randint(0, len(self.zeichen) - 1)]
        return resultat


class TestSuche:
    """
    Fuehrt die gewuenschten Laufzeittests durch.
    """

    def __init__(self, anzahl):
        """
        Legt die benoetigten Objekte an und besetzt die Attributwerte.
        :param anzahl Anzahl der Datenelemente in Liste und / oder Baum
        """
        self.anzahl = anzahl
        self.l = Liste()
        self.b = Baum()
        self.g = Generator(10)
        self.zufall = random.Random()
        self.schluesselwerte = []
        for i in range(anzahl):
            self.schluesselwerte.append(self.g.BezeichnerGenerieren())

    def TestWaehlen(self):
        """
        Zu suchendes Testelement auswaehlen.
        :return Testelement
        """
        return self.schluesselwerte[self.zufall.randint(0, self.anzahl - 1)]

    def ZeitFuerListenSucheMessen(self, d):
        """
        Misst die Zeit fuer einen Suchvorgang in der Liste.
        :param d das zu suchende Datenelement
        :return die benoetigte Suchzeit
        """
        start = time.perf_counter_ns()
        self.l.Suchen(d)
        ende = time.perf_counter_ns()
        return ende - start

    def ZeitFuerBaumSucheMessen(self, d):
        """
        Misst die Zeit fuer einen Suchvorgang im Baum.
        :param d das zu suchende Datenelement
        :return die benoetigte Suchzeit
        """
        start = time.perf_counter_ns()
        self.b.Suchen(d)
        ende = time.perf_counter_ns()
        return ende - start

    def DurchschnittsTest(self):
        """
        Bestimmt die durschnittlichen Suchzeiten.
        Es werden 5000 Suchvorange ausgefuehrt, um einen angemessenen
        Durchschnitt zu erhalten
        """
        for d in self.schluesselwerte:
            self.l.Einfuegen(d)
            self.b.Einfuegen(d)
        self.summe1 = 0
        self.summe2 = 0
        for i in range(5000):
            d = self.TestWaehlen()
            self.summe1 += self.ZeitFuerListenSucheMessen(d)
            self.summe2 += self.ZeitFuerBaumSucheMessen(d)
        self.summe1 = self.summe1 // 5000
        self.summe2 = self.summe2 // 5000

    def ZeitFuerListensucheGeben(self):
        """
        Meldet die durchschnittliche Zeit fuer die Suche in der Liste.
        :return Suchzeit
        """
        return self.summe1

    def ZeitFuerBaumsucheGeben(self):
        """
        Meldet die durchschnittliche Zeit fuer die Suche im Baum.
        :return Suchzeit
        """
        return self.summe2


test = TestSuche(100)
test.DurchschnittsTest()
print("Mittlere Suchzeit in der Liste [ns]:", test.ZeitFuerListensucheGeben())
print("Mittlere Suchzeit im Baum      [ns]:", test.ZeitFuerBaumsucheGeben())