from helferklassen import Knoten, Kante, Lesen

import random

class Graph:
    """
    Verwaltet einen ungerichteten, gewichteten Graphen mittels Adjazenzmatrix

    @author Johannes Neumeyer, Albert Wiedemann
    @version 1.0
    """

    def __init__(self):
        """
        Baut die Datenstruktur auf
        """
        self.__knoten = []
        self.__kanten = []
        self.__matrix = []

    def KnotenEinfuegen(self, bezeichner, x, y):
        """
        Einfügen eines neuen Knoten in den Graphen
        @param bezeichner Bezeichner des neuen Knotens, der dem Graphen
               hinzugefügt wird.
        @param x x-Koordinate für die Anzeige des Knotens
        @param y y-Koordinate für die Anzeige des Knotens
        """
        self.__knoten.append(Knoten(bezeichner, x, y))
        for zeile in self.__matrix:
            zeile.append(-1)
        self.__matrix.append([])
        for _ in range(len(self.__knoten)):
            self.__matrix[-1].append(-1)

    def __KnotenGeben(self, bezeichner):
        """
        Gibt den Knoten mit dem angegebenen Bezeichner zurück.
        Wenn ein Knoten mit diesem Bezeichner nicht bekannt ist, wird None
        zurückgegeben
        @param bezeichner Bezeichner des Knoten der gesucht wird
        @return Referenz auf das Knotenobjekt oder None
        """
        for k in self.__knoten:
            if k.BezeichnerGeben() == bezeichner:
                return k
        return None

    def KnotenNummerGeben(self, bezeichner):
        """
        Gibt die interne Nummer des Knotens
        Wenn ein Knoten mit diesem Bezeichner nicht bekannt ist wird -1
        zurückgegeben
        @param bezeichner Bezeichner des Knoten der gesucht wird
        @return Indexnummer des Knotens im Knotenfeld; 0 <= res <= len(knoten)-1
                bzw. -1
        """
        for i in range(len(self.__knoten)):
            if self.__knoten[i].BezeichnerGeben() == bezeichner:
                return i
        return -1

    def KnotenBezeichnerGeben(self, knoten_nummer):
        """
        Gibt die Bezeichnung eines Knotens mit der internen Knotennummer
        @param knoten_nummer Indexnummer des Knotens im Knotenarray; 0<=x<knoten.size()
        @return Bezeichner des Knotens
        """
        if 0 <= knoten_nummer < len(self.__knoten):
            return self.__knoten[knoten_nummer].BezeichnerGeben()
        else:
            return ""

    def KanteEinfuegen(self, von, nach, gewichtung):
        """
        Einfügen einer Kante in den Graphen
        Eine Kante ist durch einen Anfangsknoten und einen Endknoten
        festgelegt und hat eine Gewichtung
        @param von Bezeichner des Anfangsknotens
        @param nach Bezeichner des Endknotens
        @param gewichtung Gewichtung der Kante als Ganzzahl
        """
        von_nummer = self.KnotenNummerGeben(von)
        nach_nummer = self.KnotenNummerGeben(nach)
        if (von_nummer != -1 and nach_nummer != -1 and von_nummer != nach_nummer):
            self.__matrix[von_nummer][nach_nummer] = gewichtung
            self.__matrix[nach_nummer][von_nummer] = gewichtung
            # Es koennte geprueft werden, ob die Kante schon existiert
            neue_kante = Kante(self.__knoten[von_nummer], self.__knoten[nach_nummer], gewichtung)
            self.__kanten.append(neue_kante)

    def Ausgeben(self):
        """
        Gibt die Adjazenzmatrix des Graphen in der Konsole aus
        Nach Zeilen und Spalten formatiert
        Als Spaltenbreite wurde hier 4 Zeichen gewählt.
        """
        # Kopfzeile
        print("    ", end="")
        for i in range(len(self.__knoten)):
            print((self.__knoten[i].BezeichnerGeben() + "    ")[:4], end="")
        print()
        for i in range(len(self.__knoten)):
            print((self.__knoten[i].BezeichnerGeben() + "    ")[:4], end="")
            for j in range(len(self.__knoten)):
                if self.__matrix[i][j] != -1:
                    print((str(self.__matrix[i][j]) + "   ")[:4], end="")
                else:
                    print("    ", end="")
            print("")

    def KnotenAnzahlgeben(self):
        """
        Gibt die Anzahl der Knoten des Graphen zurück
        @return Anzahl der Knoten
        """
        return len(self.__knoten)

    def KanteGewichtGeben(self, von, nach):
        """
        Gibt die Gewichtung einer Kante zurück
        Die Kante ist durch einen Anfangsknoten und einen Endknoten
        festgelegt
        @param von Bezeichner des Anfangsknotens
        @param nach Bezeichner des Endknotens
        @return Gewichtung der Kante
        """
        von_nummer = self.KnotenNummerGeben(von)
        nach_nummer = self.KnotenNummerGeben(nach)
        if von_nummer != -1 and nach_nummer != -1:
            return self.__matrix[von_nummer][nach_nummer]
        else:
            return -1

    def ZurueckSetzen(self):
        """
        Löscht die Kanten und Knoten des Graphen.
        """
        for k in self.__knoten:
            k.SymbolGeben().Entfernen()
        for k in self.__kanten:
            k.KantensymbolGeben().Entfernen()
        self.__knoten.clear()
        self.__kanten.clear()
        self.__matrix.clear()

    def __Finden(self, knoten):
        """
        Sucht rekursiv die Wurzel des Teilbaums, zu dem der gegebene Knoten
        gehört. Wird im Rahmen des Union-Find-Algorithmus genutzt.
        @param knoten Der Knoten, dessen Wurzel gesucht wird.
        @return Die Wurzel des zugehörigen Teilbaums.
        """
        if knoten.VorgaengerGeben() is None:
            return knoten
        else:
            return self.__Finden(knoten.VorgaengerGeben())

    def __Vereinigen(self, k1, k2):
        """
        Vereinigt die Teilbäume, zu denen die beiden übergebenen Knoten
        gehören, indem der Wurzelknoten des zweiten Baums auf den
        Wurzelknoten des ersten zeigt.
        @param k1 ein Knoten aus dem ersten Teilbaum
        @param k2 ein Knoten aus dem zweiten Teilbaum
        """
        wurzel_k1 = self.__Finden(k1)
        wurzel_k2 = self.__Finden(k2)
        wurzel_k2.VorgaengerSetzen(wurzel_k1)

    def __MatrixZuKantenliste(self):
        """
        Erzeugt eine Kopie der zentral verwalteten Kantenliste.
        @return Liste aller im Graphen vorhandenen Kanten
        """
        return list(self.__kanten)

    def __PreOrderGeben(self, min_spannbaum):
        """
        Führt eine Preorder-Traversierung durch, beginnend bei einem zufällig
        ausgewählten Wurzelknoten des minimalen Spannbaums, und gibt die
        besuchte Knotenfolge als Liste zurück.
        @param min_spannbaum Kanten des minimalen Spannbaums
        @return Liste der Knoten in Preorder-Reihenfolge
        """
        self.__MinSpannbaumZuBaum(min_spannbaum)
        zufallsindex = random.randint(0, len(min_spannbaum) - 1)
        wurzel = min_spannbaum[zufallsindex].StartGeben()
        wurzel.FarbeSetzen("rot")
        pre_order_folge = []
        return self.__PreOrderRekursiv(wurzel, pre_order_folge)

    def __PreOrderRekursiv(self, wurzel, pre_order_folge):
        """
        Rekursive Hilfsmethode für Preorder-Traversierungen auf Bäumen.
        Fügt die Knoten in Preorder-Reihenfolge zur übergebenen Liste hinzu.
        @param wurzel Der aktuelle Wurzelknoten
        @param pre_order_folge bisher ermittelte Reihenfolge der Knoten
        @return aktualisierte Preorder-Knotenfolge
        """
        pre_order_folge_lokal = list(pre_order_folge)
        pre_order_folge_lokal.append(wurzel)
        wurzel.DfsBesuchtSetzen()
        baumnachbarn = wurzel.BaumnachbarnGeben()

        for k in baumnachbarn:
            if k.StartGeben() == wurzel and not k.ZielGeben().DfsBesuchtGeben():
                pre_order_folge_lokal = self.__PreOrderRekursiv(
                    k.ZielGeben(), pre_order_folge_lokal)
            elif k.ZielGeben() == wurzel and not k.StartGeben().DfsBesuchtGeben():
                pre_order_folge_lokal = self.__PreOrderRekursiv(
                    k.StartGeben(), pre_order_folge_lokal)
        return pre_order_folge_lokal

    def __MinSpannbaumZuBaum(self, min_spannbaum):
        """
        Initialisiert die Baum-Nachbarschaften der Knoten entsprechend dem
        minimalen Spannbaum. Fügt die entsprechenden Kanten als Baumkanten
        zu den Knoten hinzu.
        @param min_spannbaum Kanten des minimalen Spannbaums
        """
        for kante in min_spannbaum:
            kante.StartGeben().BaumnachbarSetzen(kante)
            kante.ZielGeben().BaumnachbarSetzen(kante)


class Beispiel:
    """
    Beispiele für diverse Graphen

    @author Albert Wiedemann, Johannes Neumeyer
    @version 1.0
    """

    def __init__(self):
        """
        Legt das Graphenobjekt an.
        """
        self.g = Graph()
        self.__l = Lesen()

    def LehrtextgraphAnzeigen(self):
        self.g.ZurueckSetzen()
        if self.__l.LesenDatenbank("Handlungsreise_Lehrtext.grdb", self.g):
            self.g.Ausgeben()

    def DeutschlandgraphAnzeigen(self):
        self.g.ZurueckSetzen()
        if self.__l.LesenDatenbank("Handlungsreise_Deutschland.grdb", self.g):
            self.g.Ausgeben()

    # def NearestNeighbour_Lehrtext(self, start):
        # self.g.ZurueckSetzen()
        # if self.__l.LesenDatenbank("Handlungsreise_Lehrtext.grdb", self.g):
            # self.g.Ausgeben()
            # self.g.NearestNeighbour(start)

    # def NearestNeighbour_Deutschland(self, start):
        # self.g.ZurueckSetzen()
        # if self.__l.LesenDatenbank("Handlungsreise_Deutschland.grdb", self.g):
            # self.g.Ausgeben()
            # self.g.NearestNeighbour(start)

    # def Kruskal_Lehrtext(self):
        # self.g.ZurueckSetzen()
        # if self.__l.LesenDatenbank("Handlungsreise_Lehrtext.grdb", self.g):
            # self.g.Ausgeben()
            # self.g.ApproximationMitKruskal()

    # def Kruskal_Deutschland(self):
        # self.g.ZurueckSetzen()
        # if self.__l.LesenDatenbank("Handlungsreise_Deutschland.grdb", self.g):
            # self.g.Ausgeben()
            # self.g.ApproximationMitKruskal()

b = Beispiel()
b.DeutschlandgraphAnzeigen()