from graphics_and_games_klassen import Kreis, Text, Rechteck, Dreieck
import math, sqlite3

class KnotenSymbol:
    """
    Verwaltet ein Knotensymbol in einem Graph

    @author Albert Wiedemann
    @version 1.0
    """

    def __init__(self, x, y, radius, farbe, bezeichner):
        """
        Legt das Symbol an und besetzt die Attribute.
        @param x x-Koordinate
        @param y y-Koordinate
        @param radius Radius
        @param farbe Farbe
        @param bezeichner Bezeichner
        """
        self.__x = x
        self.__y = y
        self.__r = radius
        self.__farbe = farbe
        self.__bezeichner = bezeichner
        self.__aussen = Kreis()
        self.__aussen.FarbeSetzen("schwarz")
        self.__innen = Kreis()
        self.__text = Text()
        self.__text.TextGroesseSetzen(18)
        self.__DarstellungAktualisieren()

    def __DarstellungAktualisieren(self):
        """
        Aktualisiert das Symbol
        """
        self.__aussen.PositionSetzen(self.__x, self.__y)
        self.__aussen.RadiusSetzen(self.__r)
        self.__innen.PositionSetzen(self.__x, self.__y)
        self.__innen.RadiusSetzen(self.__r - 2)
        self.__innen.FarbeSetzen(self.__farbe)
        self.__text.PositionSetzen(
            int(self.__x - len(self.__bezeichner) / 2 * 10 -
                len(self.__bezeichner) % 2 * 5),
            self.__y + 6)
        self.__text.TextSetzen(self.__bezeichner)

    def FarbeSetzen(self, f):
        """
        Setzt die Farbe der Darstellung
        @param f die (neue) Farbe
        """
        self.__farbe = f
        self.__DarstellungAktualisieren()

    def PositionSetzen(self, x, y):
        """
        Setzt die Position der Darstellung
        @param x x-Koordinate
        @param y y-Koordinate
        """
        self.__x = x
        self.__y = y
        self.__DarstellungAktualisieren()

    def BezeichnerSetzen(self, bezeichner):
        """
        Setzt die Farbe der Darstellung
        @param bezeichner der (neue) Bezeichner
        """
        self.__bezeichner = bezeichner
        self.__DarstellungAktualisieren()

    def XGeben(self):
        """
        Meldet die x-Koordinate des Symbols zurück.
        @return x-Koordinate
        """
        return self.__x

    def YGeben(self):
        """
        Meldet die y-Koordinate des Symbols zurück.
        @return y-Koordinate
        """
        return self.__y

    def BezeichnerGeben(self):
        """
        Meldet den Bezeichner des Symbols zurück.
        @return Bezeichner
        """
        return self.__bezeichner

    def Entfernen(self):
        """
        Entfernt das Knotensymbol aus der Anzeige
        """
        self.__aussen.Entfernen()
        self.__innen.Entfernen()
        self.__text.Entfernen()


class KantenSymbol:
    """
    Verwaltet ein Kantensymbol in einem Graph

    @author Albert Wiedemann
    @version 1.0
    """

    def __init__(self, start, ende, gerichtet, gewicht, breite, farbe):
        """
        Legt das Symbol an und besetzt die Attribute.
        @param start: Startknoten
        @param ende: Endknoten
        @param gerichtet: wenn wahr, ist der Weg gerichtet
        @param gewicht: Kantengewicht
        @param breite: Breite der Linie
        @param farbe: Farbe der Linie
        """
        self.__von = start
        self.__nach = ende
        self.__farbe = farbe
        self.__breite = breite
        self.__gerichtet = gerichtet
        self.__gewicht = gewicht
        self.__aussen = Rechteck()
        self.__innen = Rechteck()
        self.__pfeil = Dreieck()
        self.__pfeil.FarbeSetzen("schwarz")
        self.__text = Text()
        self.__text.SichtbarkeitSetzen(gewicht != "")
        self.__text.TextSetzen(gewicht)
        self.DarstellungAktualisieren()

    def DarstellungAktualisieren(self):
        """
        Aktualisiert das Symbol
        """
        x1 = self.__von.XGeben()
        y1 = self.__von.YGeben()
        x2 = self.__nach.XGeben()
        y2 = self.__nach.YGeben()
        laenge = int(math.sqrt((x1 - x2) * (x1 - x2) +
                              (y1 - y2) * (y1 - y2)))
        xm = int((x1 + x2) / 2.0)
        ym = int((y1 + y2) / 2.0)
        x = int(xm - laenge / 2)
        y = int(ym - self.__breite / 2)
        self.__aussen.PositionSetzen(x, y)
        self.__aussen.GroesseSetzen(laenge, self.__breite)
        if self.__breite < 4:
            groesse = self.__breite * 4
            self.__innen.SichtbarkeitSetzen(False)
            self.__aussen.FarbeSetzen(self.__farbe)
            self.__pfeil.SichtbarkeitSetzen(self.__gerichtet)
            self.__pfeil.PositionSetzen(xm, int(ym - groesse / 2))
            self.__pfeil.GroesseSetzen(groesse, groesse)
            self.__pfeil.WinkelSetzen(270)
        else:
            delta = 2 if self.__breite <= 8 else 4
            self.__innen.SichtbarkeitSetzen(True)
            self.__innen.PositionSetzen(x, int(y + delta / 2))
            self.__innen.GroesseSetzen(laenge, self.__breite - delta)
            self.__innen.FarbeSetzen(self.__farbe)
            self.__aussen.FarbeSetzen("schwarz")
            self.__pfeil.SichtbarkeitSetzen(self.__gerichtet)
            self.__pfeil.PositionSetzen(xm, int(ym - self.__breite / 2))
            self.__pfeil.GroesseSetzen(self.__breite - delta, self.__breite)
            self.__pfeil.WinkelSetzen(270)

        winkel = 0
        if x1 == x2:
            if y2 > y1:
                winkel = -90
            else:
                winkel = 90
        elif x1 < x2:
            winkel = -math.atan((y2 - y1) / (x2 - x1)) / math.pi * 180.0
        else:
            winkel = (180 -
                      math.atan((y2 - y1) / (x2 - x1)) / math.pi * 180.0)
        self.__pfeil.WinkelSetzen(int(winkel + 270))
        self.__innen.WinkelSetzen(int(winkel))
        self.__aussen.WinkelSetzen(int(winkel))
        self.__pfeil.GanzNachHintenBringen()
        self.__innen.GanzNachHintenBringen()
        self.__aussen.GanzNachHintenBringen()
        if (-90 <= winkel) and (winkel <= 0):
            w = (winkel + 90) / 180.0 * math.pi
            delta = self.__breite
            self.__text.PositionSetzen(xm + int(delta * math.cos(w)),
                                       ym - int(delta * math.sin(w)))
        elif (0 < winkel) and (winkel <= 90):
            w = (winkel + 90) / 180.0 * math.pi
            delta = self.__breite
            self.__text.PositionSetzen(xm - int(delta * math.cos(w)),
                                       ym + int(delta * math.sin(w)) + 7)
        elif (90 < winkel) and (winkel <= 180):
            w = (winkel + 90) / 180.0 * math.pi
            delta = self.__breite
            self.__text.PositionSetzen(xm - int(delta * math.cos(w)),
                                       ym + int(delta * math.sin(w)))
        else:
            w = (winkel + 90) / 180.0 * math.pi
            delta = self.__breite
            self.__text.PositionSetzen(xm + int(delta * math.cos(w)),
                                       ym - int(delta * math.sin(w)) + 10)

    def Entfernen(self):
        """
        Entfernt das Kantensymbol aus der Anzeige
        """
        self.__aussen.Entfernen()
        self.__innen.Entfernen()
        self.__pfeil.Entfernen()
        self.__text.Entfernen()

    def FarbeSetzen(self, f):
        """
        Setzt die Farbe der Darstellung
        @param f die (neue) Farbe
        """
        self.__farbe = f
        self.DarstellungAktualisieren()

    def StartsymbolGeben(self):
        """
        Meldet den Startknoten
        @return Startknoten
        """
        return self.__von

    def ZielsymbolGeben(self):
        """
        Meldet den Zielknoten
        @return Zielknoten
        """
        return self.__nach

    def TextSichtbarkeitSetzen(self, sichtbar):
        self.__text.SichtbarkeitSetzen(sichtbar)

    def TextEntfernen(self):
        self.__text.Entfernen()
        self.DarstellungAktualisieren()


class Knoten:
    """
    Beschreibt einen Knoten

    @author Albert Wiedemann
    @version 1.0
    """

    def __init__(self, bezeichner, x, y):
        """
        Besetzt die Attribute und legt das Knotensymbol an.
        @param bezeichner Bezeichner
        @param x x-Koordinate
        @param y y-Koordinate
        """
        self.__bezeichner = bezeichner
        self.__laenge = 0
        self.__vorgaenger = None
        self.__symbol = KnotenSymbol(x, y, 20, "weiss", bezeichner)
        self.__dfsBesucht = False
        self.__baumnachbarn = []

    def BezeichnerGeben(self):
        """
        Meldet den Bezeichner des Knotens zurück.
        @return Bezeichner
        """
        return self.__bezeichner

    def SymbolGeben(self):
        """
        Meldet das Symbol des Knotens zurück.
        @return Symbol
        """
        return self.__symbol

    def FarbeSetzen(self, f):
        """
        Setzt die Farbe der Darstellung
        @param f die (neue) Farbe
        """
        self.__symbol.FarbeSetzen(f)

    def LaengeSetzen(self, l):
        """
        Setzt die Weglaenge
        @param l die (neue) Laenge
        """
        self.__laenge = l

    def LaengeGeben(self):
        """
        Meldet die Weglaenge zurück.
        @return Weglaenge
        """
        return self.__laenge

    def VorgaengerSetzen(self, v):
        """
        Setzt den Vorgaenger
        @param v der Vorgaenger
        """
        self.__vorgaenger = v

    def VorgaengerGeben(self):
        """
        Meldet den Vorgaenger zurück.
        @return Vorgaenger
        """
        return self.__vorgaenger

    def BaumnachbarSetzen(self, n):
        """
        Fuegt einen Nachbarknoten hinzu
        @param n Kante zum Nachbarknoten
        """
        self.__baumnachbarn.append(n)

    def BaumnachbarnGeben(self):
        """
        Gibt das Feld der Kanten zu den Nachbarknoten zurück
        @return Kanten zu den Nachbarknoten
        """
        return self.__baumnachbarn

    def DfsBesuchtGeben(self):
        """
        Meldet den Wert der Variablen dsfBesucht
        @return dsfBesucht
        """
        return self.__dfsBesucht

    def DfsBesuchtSetzen(self):
        """
        Setzt den Wert der Variablen dsfBesucht auf wahr (besucht)
        """
        self.__dfsBesucht = True

    def DfsUnbesuchtSetzen(self):
        """
        Setzt den Wert der Variablen dsfBesucht auf falsch (nicht besucht)
        """
        self.__dfsBesucht = False


class Kante:
    """
    Kantenbeschreibung für die Konstruktion des minimalen Spannbaums

    @author Johannes Neumeyer
    @version 1.0
    """

    def __init__(self, start, ziel, laenge):
        """
        Besetzt die Attribute
        """
        self.__start = start
        self.__ziel = ziel
        self.__laenge = laenge
        self.__symbol = KantenSymbol(start.SymbolGeben(), ziel.SymbolGeben(),
                                   False, str(laenge), 3, "blau")

    def FarbeSetzen(self, f):
        """
        Setzt die Farbe der Darstellung
        @param f die (neue) Farbe
        """
        self.__symbol.FarbeSetzen(f)

    def StartGeben(self):
        """
        Meldet den Startknoten
        @return Startknoten
        """
        return self.__start

    def ZielGeben(self):
        """
        Meldet den Zielknoten
        @return Zielknoten
        """
        return self.__ziel

    def LaengeGeben(self):
        """
        Meldet die Laenge der Kante
        @return Laenge
        """
        return self.__laenge

    def KantensymbolGeben(self):
        """
        Meldet das Kantensymbol
        @return Kantensymbol
        """
        return self.__symbol

class Lesen:
    """
    Stellt Methoden zum Lesen der Graphendaten bereit
    """


    def LesenDatenbank(self, name, g):
        """
        Liest die Datenbank unter dem gegebenen Namen ein und trägt die Daten in den Graphen g ein.        
        """
        try:
            verbindung = sqlite3.connect(name)
            daten = verbindung.execute("SELECT bezeichner, x, y FROM knoten;")
            for datensatz in daten:
                g.KnotenEinfuegen(str(datensatz[0]), int(datensatz[1]), int(datensatz[2]))

            daten = verbindung.execute("SELECT bezeichnerStart, bezeichnerZiel, gewicht, gerichtet FROM kanten;")
            for datensatz in daten:
                g.KanteEinfuegen(str(datensatz[0]), str(datensatz[1]), int(datensatz[2]))

            verbindung.close()
            return True
        except Exception as e:
            print(e)
            return False

    def LesenDatei(self, name, g):
        """
        Liest die Datei unter dem gegebenen Namen ein und trägt die Daten in den Graphen g ein.
        """
        try:
            with open(name, 'r') as datei:
                zeilen = [zeile.rstrip() for zeile in datei]

                anzahl = int(zeilen[0].split(": ")[1])
                aktuelle_zeile = 1
                for i in range(anzahl):
                    teile = zeilen[aktuelle_zeile].split("\t")
                    g.KnotenEinfuegen(teile[2], int(teile[0]), int(teile[1]))
                    aktuelle_zeile += 1
                anzahl = int(zeilen[aktuelle_zeile].split(": ")[1])
                aktuelle_zeile += 1
                for i in range(anzahl):
                    teile = zeilen[aktuelle_zeile].split("\t")
                    g.KanteEinfuegen(teile[0], teile[1], int(teile[2]))
                    aktuelle_zeile += 1
            return True
        except Exception as e:
            print(e)
            return False
