import UIKit
import PlaygroundSupport

/**
Generiert den Graphen (Knotenfeld und Adjazenzmatrix).

 * @author Albert Wiedemann
 * @version 1.0
 */
class Generieren
{
    /** Maximale Entfernung zweier Knoten */
    private let maxEntfernung = 100
    /** Adjazenzmatrix */
    private var matrix: [[Int]]

    /**
    Legt die Felder an.
    - parameters:
        - anzahl: Anzahl der Knoten
     */
    init(anzahl: Int)
    {
        matrix = [[Int]](repeating: [Int](repeating: -1, count: anzahl), count: anzahl)
    }
    
    /**
    Generiert die Adjazenzmatrix.
    Auf eine zweidimensionale Darstellbarkeit wird nicht geachtet.
    - parameters:
        - anzahl: Anzahl der zu generierenden Kanten
    - returns: Adjazenzmatrix
     */
    func AdjazenzmatrixGenerieren(anzahl: Int) -> [[Int]]
    {
        for i in 0 ..< matrix.count
        {
            matrix[i][i] = 0
        }
        for _ in 0 ..< anzahl
        {
            var von: Int, nach: Int, weg: Int
            repeat
            {
                von = Int.random(in: 0 ..< matrix.count)
                nach = Int.random(in: 0 ..< matrix.count)
            }
            while matrix[von][nach] >= 0
            weg = Int.random(in: 0 ..< maxEntfernung)
            matrix[von][nach] = weg
            matrix[nach][von] = weg
        }
        return matrix
    }
}

/**
Knoteninformation für Dijkstra-Algorithmus.

- author: Albert Wiedemann
- version: 1.0
 */
class Knoten
{
    /** Knotennummer */
    var nummer: Int
    /** Weglänge bis zu diesem Knoten */
    private var länge: Int

    /**
    Belegt die Attribute vor.
    - parameterts:
        - n: Nummer des Knotens
    */
    init(n: Int)
    {
        nummer = n
        länge = 0
    }
    
    /**
    Meldet die Nummer des Knotens.
    - returns: Knotennummer
     ´*/
    func NummerGeben() -> Int
    {
        return nummer
    }
    
    /**
    Setzt die Weglänge.
    - parameters:
        - längeNeu: neue Weglänge
    */
    func LängeSetzen(längeNeu: Int)
    {
        länge = längeNeu
    }
    
    /**
    Meldet die Weglänge.
    - returns: die Weglänge
     */
    func LängeGeben() -> Int
    {
        return länge
    }
}

/**
Sucht im Graph nach dem Alorithmus von Dijkstra.

- author: Albert Wiedemann
- version: 1.0
 */
class Dijkstra
{
    /** Adjazenzmatrix des Graphen */
    private var matrix: [[Int]]
    /** Liste der abgearbeiteten Knoten */
    private var fertig: [Knoten]
    /** Liste der unbearbeiteten Knoten */
    private var rest: [Knoten]
    /** Liste der erreichten Knoten */
    private var inarbeit: [Knoten]

    /**
    Legt die Hilfsdaten an.
    - parameters:
        - m: Adjazenzmatrix
    */
    init(m: [[Int]])
    {
        matrix = m
        fertig = []
        rest = []
        inarbeit = []
    }

    /**
    Führt die Suche aus.
    - parameters:
        - von: Startknoten
        - nach: Zielknoten
    - returns: Weglänge
    */
    func WegSuchen(von: Int, nach: Int) -> Int
    {
        for i in 0 ..< matrix.count
        {
            if i == von
            {
                fertig.append(Knoten(n: i))
            }
            else
            {
                inarbeit.append(Knoten(n: i))
            }
        }
        var akt = von
        var länge = 0
        while akt != nach
        {
            for i in 0 ..< matrix.count
            {
                if matrix[akt][i] > 0
                {
                    if let k = KnotenSuchen(feld: rest, nummer: i)
                    {
                        k.LängeSetzen (längeNeu: länge + matrix[akt][i])
                        rest.remove(at: rest.firstIndex(where: { $0 === k})!)
                        inarbeit.append(k)
                    }
                    else
                    {
                        if let k = KnotenSuchen(feld: inarbeit, nummer: i)
                        {
                            if länge + matrix[akt][i] < k.LängeGeben ()
                            {
                                k.LängeSetzen(längeNeu: länge + matrix[akt][i])
                            }
                        }
                    }
                }
            }
            var nächster: Knoten? = nil
            for k in inarbeit
            {
                if (nächster == nil) || (nächster!.LängeGeben() > k.LängeGeben())
                {
                    nächster = k
                }
            }
            if nächster == nil
            {
                return -1
            }
            inarbeit.remove(at: inarbeit.firstIndex(where: { $0 === nächster!})!)
            fertig.append(nächster!)
            länge = nächster!.LängeGeben();
            akt = nächster!.NummerGeben();
        }
        return länge
    }
    
    /**
    Sucht den Knoten mit der angegebenen Nummer
    - parameters:
        - feld: das Knotenfeld, in dem gesucht werden soll
        - nummer: die Nummer des gesuchten Knotens
     * @return Knotenreferenz oder null, falls der Knoten mit der angegebenen Nummer nicht im Feld ist.
     */
    private func KnotenSuchen(feld: [Knoten], nummer: Int) -> Knoten?
    {
        for k in feld
        {
            if k.NummerGeben() == nummer
            {
                return k
            }
        }
        return nil
    }
}

/**
Führt die gewünschten Laufzeittests durch.

- author: Albert Wiedemann
- version: 1.0
 */
class TestSuchen
{
    /** Anzahl der Datenelemente im Graphen */
    private var anzahl: Int
    /** Die Laufzeitsumme */
    private var zeitDijkstra: TimeInterval
    /** Die Adjazenzmatrix */
    private var matrix: [[Int]]
    /** Startort */
    private var start: Int
    /** Zielort */
    private var ziel: Int

    /**
    Legt die benötigten Objekte an und besetzt die Attributwerte.
    - parameters:
        - anzahl: Anzahl der Knoten im Graph
     */
    init(anzahl: Int)
    {
        self.anzahl = anzahl
        start = -1
        ziel = -1
        zeitDijkstra = 0
        matrix = Generieren(anzahl: anzahl).AdjazenzmatrixGenerieren(anzahl: (anzahl - 1) * 2)
    }
   
    /**
    Bestimmt die durschnittliche Zeit für die Knotenbesuche.
    */
    func DurchschnittsTest()
    {
        zeitDijkstra = 0
        for _ in 0 ..< 100
        {
            start = Int.random(in: 0 ..< anzahl)
            repeat
            {
                ziel = Int.random(in: 0 ..< anzahl)
            }
            while (start ==  ziel)
            let t = Dijkstra(m: matrix)
            let startZeit = Date()
                    t.WegSuchen(von: start, nach: ziel)
            zeitDijkstra += Date().timeIntervalSince(startZeit)
        }
        zeitDijkstra /= 100
    }
    
    /**
    Meldet die durchschnittliche Zeit für die Suche nach Dijkstra.
    - returns: Suchzeit
     */
    func ZeitFürDijkstraGeben() -> Int
    {
        return Int(zeitDijkstra * 1000000)
    }
}

/**
 Verwaltet die Viewklassen
 */
class MyViewController : UIViewController
{
    /** Eingabefeld für die Knotenanzahl */
    let anzahlFeld = UITextField()
    /** Ausgabefeld für die Zeit */
    let zeitFeld = UILabel()

    override func viewDidLoad()
    {
        view.backgroundColor = UIColor(red: 0.95, green: 0.95, blue: 0.95, alpha: 1.0)

        let label1 = UILabel()
        label1.text = "Anzahl der Knoten im Graph"
        label1.textColor = .black
        label1.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(label1)
        let label2 = UILabel()
        label2.text = "Zeit für die Wegsuche nach Dijkstra [µs]"
        label2.textColor = .black
        label2.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(label2)
        anzahlFeld.backgroundColor = .white
        anzahlFeld.placeholder = "Anzahl der Knoten"
        anzahlFeld.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(anzahlFeld)
        zeitFeld.text = "---"
        zeitFeld.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(zeitFeld)
        let test = UIButton(type: .system)
        test.setTitle("Test ausführen", for: .normal)
        test.layer.cornerRadius = 10
        test.backgroundColor = .white
        test.addTarget(self, action: #selector(Ausführen), for: .touchUpInside)
        test.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(test)

        NSLayoutConstraint.activate([
            label1.topAnchor.constraint(equalTo: view.topAnchor, constant: 16),
            label1.leadingAnchor.constraint(equalTo: view.leadingAnchor, constant: 16),
            label1.widthAnchor.constraint(greaterThanOrEqualToConstant: 350),
            label2.topAnchor.constraint(equalTo: label1.bottomAnchor, constant: 16),
            label2.leadingAnchor.constraint(equalTo: view.leadingAnchor, constant: 16),
            label2.widthAnchor.constraint(greaterThanOrEqualToConstant: 350),
            anzahlFeld.leadingAnchor.constraint(equalTo: label1.trailingAnchor),
            anzahlFeld.centerYAnchor.constraint(equalTo: label1.centerYAnchor),
            zeitFeld.leadingAnchor.constraint(equalTo: label2.trailingAnchor),
            zeitFeld.centerYAnchor.constraint(equalTo: label2.centerYAnchor),
            test.leadingAnchor.constraint(equalTo: view.leadingAnchor, constant: 266),
            test.topAnchor.constraint(equalTo: label2.bottomAnchor, constant: 16),
            test.widthAnchor.constraint(greaterThanOrEqualToConstant: 200)
        ])
        
    }
    
    /**
     Aktionsmethode für den Ausführen-Knopf
     */
    @objc func Ausführen()
    {
        zeitFeld.text = "---"
        if let anzahl = Int(anzahlFeld.text!)
        {
            let test = TestSuchen(anzahl: anzahl)
            test.DurchschnittsTest()
            zeitFeld.text = "\(test.ZeitFürDijkstraGeben())"
        }
        else
        {
            anzahlFeld.text = "Falsche Eingabe"
        }
    }
}
// Present the view controller in the Live View window
PlaygroundPage.current.liveView = MyViewController()
