import UIKit
import PlaygroundSupport

/**
Generiert zufällige Bezeichner vom Typ String gegebener Länge.

 - author: Albert Wiedemann
- version: 1.0
*/
class Generator
{
    /** Der Zeichenvorrat für die Bezeichner */
    private let zeichen: [Character] = ["a", "b", "c", "d", "e", "f", "g", "h", "i", "j"]
    /** Die Länge der zu generierenden Bezeichner */
    private var länge: Int

    /**
    Initialisiert den Zufallsgenerator.
    - parameters:
        - länge: Die Länge der zu generierenden Bezeichner
     */
    init(länge: Int)
    {
        self.länge = länge
    }
    
    /**
    erzeugt einen neuen Bezeichner.
    - returns: der neue Bezeichner
    */
    func BezeichnerGenerieren() -> String
    {
        var resultat = ""
        for _ in 1 ... länge
        {
            resultat = resultat + String(zeichen[Int.random(in: 0 ..< zeichen.count)])
        }
        return resultat
    }
}

/**
Eine sortierte Liste nach Standardimplementierung.

- author: Albert Wiedemann
- version: 1.0
 */
class Liste
{
    /**
    Beschreibt ein abstraktes Element der Liste.

    - author: Albert Wiedemann
    - version: 1.0
     */
    class ListenElement
    {
        
        /**
         Keine Aufgabe, nur zur Dokumentation aufgeführt.
         */
        init()
        {
            
        }
        
        /**
         Fügt ein neues Datenelement sortiert in die Liste ein.
         - parameters:
             - neu: das neue Datenelement
         - returns: der (neue) Nachfolger für das Vorgängerelement
         */
        func Einfügen(neu: String) -> ListenElement
        {
            fatalError("ListenElement.Einfügen muss überschrieben werden.")
        }
        
        /**
         Sucht ein Datenelement mit dem in der Referenz angegebenen Schlüssel.
         - parameters:
             - schlüssel: der zu suchende Schlüsselwert
         - returns: das gefundene Datenelement oder nil
         */
        func Suchen(schlüssel: String) -> String?
        {
            fatalError("ListenElement.Suchen muss überschrieben werden.")
        }
    }


    /**
    Beschreibt ein Abschlusselement der Liste.

    - author: Albert Wiedemann
    - version: 1.0
     */
    class Abschluss: ListenElement
    {

        /**
        Für das Anlegen des Abschlusses brauchts nichts getan zu werden.
        */
        override init()
        {
            
        }

        /**
         Fügt ein neues Datenelement sortiert in die Liste ein.
         - parameters:
             - neu: das neue Datenelement
         - returns: der (neue) Nachfolger für das Vorgängerelement
         */
        override func Einfügen(neu: String) -> ListenElement
        {
            return Knoten(d: neu, nf: self)
        }
        
        /**
         Sucht ein Datenelement mit dem in der Referenz angegebenen Schlüssel.
         - parameters:
             - schlüssel: der zu suchende Schlüsselwert
         - returns: das gefundene Datenelement oder nil
         */
        override func Suchen(schlüssel: String) -> String?
        {
            return nil
        }
    }

    /**
    Beschreibt einen Knoten der Liste.

    - author: Albert Wiedemann
    - version: 1.0
     */
    class Knoten: ListenElement
    {

        /** Referenz auf das Datenelement */
        private var daten: String
        /** Referenz auf den Nachfolgerknoten */
        private var nachfolger: ListenElement
        
        /**
        Legt einen neuen Knoten mit gegebenem Datenelement und gegebenem Nachfolger an.
        - parameters:
            - d: das von diesem Knoten verwaltete Datenelement
            - nf: Referenz auf das Nachfolgerelement
         */
        init(d: String, nf: ListenElement)
        {
            daten = d
            nachfolger = nf
        }
        

        /**
         Fügt ein neues Datenelement sortiert in die Liste ein.
         - parameters:
             - neu: das neue Datenelement
         - returns: der (neue) Nachfolger für das Vorgängerelement
         */
        override func Einfügen(neu: String) -> ListenElement
        {
            if neu > daten
            {
                nachfolger = nachfolger.Einfügen(neu: neu)
                return self
            }
            else
            {
                return Knoten(d: neu, nf: self)
            }
        }

        /**
         Sucht ein Datenelement mit dem in der Referenz angegebenen Schlüssel.
         - parameters:
             - schlüssel: der zu suchende Schlüsselwert
         - returns: das gefundene Datenelement oder nil
         */
        override func Suchen(schlüssel: String) -> String?
        {
            if schlüssel == daten
            {
                return daten
            }
            else
            {
                return nachfolger.Suchen(schlüssel: schlüssel)
            }
        }
    }



    /** Referenz auf das erste Element der Liste. */
    private var anfang: ListenElement
    
    /**
    Legt eine leere Liste an.
     */
    init()
    {
        anfang = Abschluss()
    }

    /**
    Fügt ein neues Datenelement sortiert in die Liste ein.
    - parameters:
        - neu: das neue Datenelement
    */
    func Einfügen(neu: String)
    {
        anfang = anfang.Einfügen(neu: neu)
    }
    
    /**
    Sucht ein Datenelement mit dem angegebenen Schlüssel.
    - parameters:
        - schlüssel: Referenz auf ein Datenelement mit dem zu suchenden Schlüsselwert
    - returns: das gefundene Datenelement oder nil
     */
    func Suchen(schlüssel: String) -> String?
    {
        return anfang.Suchen(schlüssel: schlüssel)
    }
}

/**
Ein sortierter Binärbaum nach Standardimplementierung.

- author: Albert Wiedemann
- version: 1.0
*/
class Baum
{
    /**
    Beschreibt ein abstraktes Element des Baums.

    - author: Albert Wiedemann
    - version: 1.0
     */
    class BaumElement
    {
        
        /**
        Fügt ein neues Datenelement sortiert in den Baum ein.
        - parameters:
            - neu: das neue Datenelement
        - returns: der (neue) Nachfolger für das Vorgängerelement
        */
        func Einfügen(neu: String) -> BaumElement
        {
            fatalError("BaumElement.Einfügen muss überschrieben werden.")
        }

        /**
        Sucht ein Datenelement mit dem angegebenen Schlüssel.
        - parameters:
            - schlüssel: der zu suchende Schlüsselwert
        - returns: das gefundene Datenelement oder nil
         */
        func Suchen(schlüssel: String) -> String?
        {
            fatalError("BaumElement.Suchen muss überschrieben werden.")
        }
    }

    /**
    Beschreibt ein Abschlusselement des Baums.
     *
     * @author Albert Wiedemann
     * @version 1.0
     */
    class Abschluss: BaumElement
    {

        /**
        Für das Anlegen des Abschlusses brauchts nichts getan zu werden.
        */
        override init()
        {
            
        }

        /**
        Fügt ein neues Datenelement sortiert in den Baum ein.
        - parameters:
            - neu: das neue Datenelement
        - returns: der (neue) Nachfolger für das Vorgängerelement
        */
        override func Einfügen(neu: String) -> BaumElement
        {
            return Knoten(d: neu, lnf: self, rnf: self);
        }
        
        /**
        Sucht ein Datenelement mit dem angegebenen Schlüssel.
        - parameters:
            - schlüssel: Referenz auf ein Datenelement mit dem zu suchenden Schlüsselwert
        - returns: das gefundene Datenelement oder nil
         */
        override func Suchen(schlüssel: String) -> String?
        {
            return nil
        }
    }

    /**
    Beschreibt einen Knoten des Baums.

    - author: Albert Wiedemann
    - version: 1.0
    */
    class Knoten: BaumElement
    {
        /** das Datenelement */
        private var daten: String
        /** Referenz auf den linken Nachfolgerknoten (mit den kleineren Werten) */
        private var linksnachfolger: BaumElement
        /** Referenz auf den rechten Nachfolgerknoten (mit den größeren Werten) */
        private var rechtsnachfolger: BaumElement
        
        /**
        Legt einen neuen Knoten mit gegebenem Datenelement und gegebenen Nachfolgern an.
         - parameters:
            - d das von diesem Knoten verwaltete Datenelement
            - lnf: Referenz auf das linke Nachfolgerelement
            - rnf: Referenz auf das rechte Nachfolgerelement
         */
        init(d: String, lnf: BaumElement, rnf: BaumElement)
        {
            daten = d
            linksnachfolger = lnf
            rechtsnachfolger = rnf
        }
        
        /**
        Fügt ein neues Datenelement sortiert in den Baum ein.
        - parameters:
            - neu: das neue Datenelement
        - returns: der (neue) Nachfolger für das Vorgängerelement
        */
        override func Einfügen(neu: String) -> BaumElement
        {
            if neu > daten
            {
                rechtsnachfolger = rechtsnachfolger.Einfügen(neu: neu)
            }
            else
            {
                linksnachfolger = linksnachfolger.Einfügen(neu: neu)
            }
            return self
        }
        
        /**
        Sucht ein Datenelement mit dem angegebenen Schlüssel.
        - parameters:
            - schlüssel: Referenz auf ein Datenelement mit dem zu suchenden Schlüsselwert
        - returns: das gefundene Datenelement oder nil
         */
        override func Suchen(schlüssel: String) -> String?
        {
            if daten == schlüssel
            {
                return daten
            }
            else if schlüssel < daten
            {
                return linksnachfolger.Suchen(schlüssel: schlüssel)
            }
            else
            {
                return rechtsnachfolger.Suchen(schlüssel: schlüssel)
            }
        }
    }

    /** Referenz auf die Wurzel des Baums. */
    private var anfang: BaumElement

    /**
    Legt einen leeren Baum an.
    */
    init()
    {
        anfang = Abschluss()
    }


    /**
    Fügt ein neues Datenelement in den Baum ein.
    - parameters:
        - neu: das neue Datenelement
    */
    func Einfügen(neu: String)
    {
        anfang = anfang.Einfügen(neu: neu)
    }
    
    /**
    Sucht ein Datenelement mit dem in der Referenz angegebenen Schlüssel.
    - parameters:
        - schlüssel: der zu suchende Schlüsselwert
    - returns: das gefundene Datenelement oder nil
     */
    func Suchen(schlüssel: String) -> String?
    {
        return anfang.Suchen(schlüssel: schlüssel)
    }
}

/**
Führt die gewünschten Laufzeittests durch.

- author: Albert Wiedemann
- version: 1.0
*/
class TestSuche
{
    /** Anzahl der Datenelemente in Liste und / oder Baum */
    private var anzahl: Int
    /** Die zu testende Liste */
    private var l: Liste
    /** Der zu testende Baum */
    private var b: Baum
    /** Erzeugerelement für die Schlüsselwerte */
    private var g: Generator
    /** Die Laufzeitsummen */
    private var summe1, summe2: Double
    /** Die vorhandenen Datenelemente */
    private var schlüsselwerte: [String]

    /**
    Legt die benötigten Objekte an und besetzt die Attributwerte.
    - parameters:
        - anzahl: Anzahl der Datenelemente in Liste und / oder Baum
     */
    init(anzahl: Int)
    {
        self.anzahl = anzahl
        l = Liste()
        b = Baum()
        g = Generator(länge: 10)
        schlüsselwerte = Array<String>()
        summe1 = 0
        summe2 = 0
        for _ in 1 ... anzahl
        {
            schlüsselwerte.append(g.BezeichnerGenerieren())
        }
    }
    
    /**
    Zu suchendes Testelement auswählen.
    - returns: Testelement
    */
    private func TestWählen() -> String
    {
        return schlüsselwerte[Int.random(in: 0 ..< schlüsselwerte.count)]
    }
    
    /**
    Misst die Zeit für einen Suchvorgang in der Liste.
    - parameters:
        - d: das zu suchende Datenelement
    - returns: die benötigte Suchzeit
    */
    private func ZeitFürListenSucheMessen(d: String) -> Double
    {
        let start = Date()
        if l.Suchen(schlüssel: d) == nil
        {
            fatalError("Element in der Liste nicht gefunden")
        }
        let ende = Date()
        return ende.timeIntervalSince(start)
    }
    
    /**
    Misst die Zeit für einen Suchvorgang im Baum.
     - parameters:
         - d: das zu suchende Datenelement
     - returns: die benötigte Suchzeit
    */
    private func ZeitFürBaumSucheMessen(d: String) -> Double
    {
        let start = Date()
        if b.Suchen(schlüssel: d) == nil
        {
            fatalError("Element im Baum nihct gefunden")
        }
        let ende = Date()
        return ende.timeIntervalSince(start)
    }
    
    /**
    Bestimmt die durschnittlichen Suchzeiten.
    Es werden 5000 Suchvoränge ausgeführt, um einen angemessenen Durchschnitt zu erhalten
    */
    func DurchschnittsTest()
    {
        for d in schlüsselwerte
        {
            l.Einfügen(neu: d)
            b.Einfügen(neu: d)
        }
        summe1 = 0
        summe2 = 0
        for _ in 1 ... 5000
        {
            let d = TestWählen()
            summe1 += ZeitFürListenSucheMessen(d: d)
            summe2 += ZeitFürBaumSucheMessen(d: d)
        }
        summe1 /= 5000
        summe2 /= 5000
    }
    
    /**
    Meldet die durchschnittliche Zeit für die Suche in der Liste.
    - returns: Suchzeit
     */
    func ZeitFürListensucheGeben() -> Int
    {
        return Int(summe1 * 1000000)
    }
    
    /**
    Meldet die durchschnittliche Zeit für die Suche im Baum.
    - returns Suchzeit
     */
    func ZeitFürBaumsucheGeben() -> Int
    {
        return Int(summe2 * 1000000)
    }
}


/**
 Verwaltet die Viewklassen
 */
class MyViewController : UIViewController
{
    /** Eingabefeld für die Knotenanzahl */
    let anzahlFeld = UITextField()
    /** Ausgabefeld für die Zeit bei der Liste */
    let zeitFeldListe = UILabel()
    /** Ausgabefeld für die Zeit bei m Baum */
    let zeitFeldBaum = UILabel()

    override func viewDidLoad()
    {
        view.backgroundColor = UIColor(red: 0.95, green: 0.95, blue: 0.95, alpha: 1.0)

        let label1 = UILabel()
        label1.text = "Anzahl der Datenelemente in Liste bzw. Baum"
        label1.textColor = .black
        label1.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(label1)
        let label2 = UILabel()
        label2.text = "Mittlere Suchzeit in der Liste [µs]"
        label2.textColor = .black
        label2.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(label2)
        let label3 = UILabel()
        label3.text = "Mittlere Suchzeit im Baum [µs]"
        label3.textColor = .black
        label3.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(label3)
        anzahlFeld.backgroundColor = .white
        anzahlFeld.placeholder = "Anzahl der Knoten"
        anzahlFeld.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(anzahlFeld)
        zeitFeldListe.text = "---"
        zeitFeldListe.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(zeitFeldListe)
        zeitFeldBaum.text = "---"
        zeitFeldBaum.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(zeitFeldBaum)
        let test = UIButton(type: .system)
        test.setTitle("Test ausführen", for: .normal)
        test.layer.cornerRadius = 10
        test.backgroundColor = .white
        test.addTarget(self, action: #selector(Ausführen), for: .touchUpInside)
        test.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(test)

        NSLayoutConstraint.activate([
            label1.topAnchor.constraint(equalTo: view.topAnchor, constant: 16),
            label1.leadingAnchor.constraint(equalTo: view.leadingAnchor, constant: 16),
            label1.widthAnchor.constraint(greaterThanOrEqualToConstant: 350),
            label2.topAnchor.constraint(equalTo: label1.bottomAnchor, constant: 16),
            label2.leadingAnchor.constraint(equalTo: view.leadingAnchor, constant: 16),
            label2.widthAnchor.constraint(greaterThanOrEqualToConstant: 350),
            label3.topAnchor.constraint(equalTo: label2.bottomAnchor, constant: 16),
            label3.leadingAnchor.constraint(equalTo: view.leadingAnchor, constant: 16),
            label3.widthAnchor.constraint(greaterThanOrEqualToConstant: 350),
            anzahlFeld.leadingAnchor.constraint(equalTo: label1.trailingAnchor),
            anzahlFeld.centerYAnchor.constraint(equalTo: label1.centerYAnchor),
            zeitFeldListe.leadingAnchor.constraint(equalTo: label2.trailingAnchor),
            zeitFeldListe.centerYAnchor.constraint(equalTo: label2.centerYAnchor),
            zeitFeldBaum.leadingAnchor.constraint(equalTo: label3.trailingAnchor),
            zeitFeldBaum.centerYAnchor.constraint(equalTo: label3.centerYAnchor),
            test.leadingAnchor.constraint(equalTo: view.leadingAnchor, constant: 266),
            test.topAnchor.constraint(equalTo: label3.bottomAnchor, constant: 16),
            test.widthAnchor.constraint(greaterThanOrEqualToConstant: 200)
        ])
        
    }
    
    /**
     Aktionsmethode für den Ausführen-Knopf
     */
    @objc func Ausführen()
    {
        zeitFeldListe.text = "---"
        zeitFeldBaum.text = "---"
        if let anzahl = Int(anzahlFeld.text!)
        {
            let test = TestSuche(anzahl: anzahl)
            test.DurchschnittsTest()
            zeitFeldListe.text = "\(test.ZeitFürListensucheGeben())"
            zeitFeldBaum.text = "\(test.ZeitFürBaumsucheGeben())"
        }
        else
        {
            anzahlFeld.text = "Falsche Eingabe"
        }
    }
}
// Present the view controller in the Live View window
PlaygroundPage.current.liveView = MyViewController()

