import Foundation
import PlaygroundSupport

/**
Verwaltet ein Knotensymbol in einem Graph

- author: Albert Wiedemann
- version: 1.0
*/
class KnotenSymbol
{
    /** Außenkreis des Darstellungssymbols */
    private let außen: Kreis
    /** Innenkreis des Darstellungssymbols */
    private let innen: Kreis
    /** Text des Darstellungssymbols */
    private let text: Text
    /** Der Bezeichner */
    private var bezeichner: String
    /** Radius */
    private var r: Int
    /** x-Koordinate */
    private var x: Int
    /** y-Koordinate */
    private var y: Int
    /** Farbe */
    private var farbe: String
    
    /**
	Legt das Symbol an und besetzt die Attribute.
	- parameters:
		- x: x-Koordinate
		- y: y-Koordinate
		- radius: Radius
		- farbe: Farbe
		- bezeichner: Bezeichner
    */
    init(x: Int, y: Int, radius: Int, farbe: String, bezeichner: String)
    {
        self.x = x
        self.y = y
        self.r = radius
        self.farbe = farbe
        self.bezeichner = bezeichner
        außen = Kreis()
        außen.FarbeSetzen(farbe: "schwarz")
        innen = Kreis()
        text = Text()
        text.TextGrößeSetzen(größe: 18)
        DarstellungAktualisieren()
    }
    
    /**
    Aktualisiert das Symbol
    */
    private func DarstellungAktualisieren()
    {
        außen.PositionSetzen(x: x, y: y)
        außen.RadiusSetzen(radius: r)
        innen.PositionSetzen(x: x, y: y)
        innen.RadiusSetzen(radius: r - 2)
        innen.FarbeSetzen(farbe: farbe)
        text.PositionSetzen(x: x - bezeichner.count / 2 * 10 - bezeichner.count % 2 * 5, y: y + 6)
        text.TextSetzen(text: bezeichner)
    }
    
    /**
	Setzt die Farbe der Darstellung
	- parameters:
		- f: die (neue) Farbe
    */
    func FarbeSetzen(f: String)
    {
        farbe = f
        DarstellungAktualisieren()
    }
    
    /**
	Setzt die Position der Darstellung
	- parameters:
		- x: x-Koordinate
		- y: y-Koordinate
    */
    func PositionSetzen(x: Int, y: Int)
    {
        self.x = x
        self.y = y
        DarstellungAktualisieren()
    }
    
    /**
	Setzt die Farbe der Darstellung
	- parameters:
		- bezeichner: der (neue) Bezeichner
    */
    func BezeichnerSetzen(bezeichner: String)
    {
        self.bezeichner = bezeichner
        DarstellungAktualisieren()
    }
    
    /**
	Meldet die x-Koordinate des Symbols zurück.
	- returns: x-Koordinate
    */
    func XGeben() -> Int
    {
        return x
    }
    
    /**
	Meldet die y-Koordinate des Symbols zurück.
	- returns: y-Koordinate
     */
    func YGeben() -> Int
    {
        return y
    }
    
    /**
	Meldet den Bezeichner des Symbols zurück.
	- returns: Bezeichner
     */
    func BezeichnerGeben() -> String
    {
        return bezeichner
    }
    
    /**
    Entfernt das Knotensymbol aus der Anzeige
    */
    func Entfernen()
    {
        außen.Entfernen()
        innen.Entfernen()
        text.Entfernen()
    }
}

/**
Verwaltet ein Kantensymbol in einem Graph

- author: Albert Wiedemann
- version: 1.0
*/
class KantenSymbol
{
    /** Außenrechteck des Darstellungssymbols */
    private let außen: Rechteck
    /** Innenrechteck des Darstellungssymbols */
    private let innen: Rechteck
    /** Richtungspfeil des Darstellungssymbols */
    private let pfeil: Dreieck
    /** Gewichtsangabe des Darstellungssymbols */
    private let text: Text
    /** Breite des Darstellungssymbols */
    private var breite: Int
    /** Gerichtete Kante */
    private var gerichtet: Bool
    /** Gewicht der Kante */
    private var gewicht: String
    /** Farbe */
    private var farbe: String
    /** Startknoten */
    private var von: KnotenSymbol
    /** Endknoten */
    private var nach: KnotenSymbol

    /**
	Legt das Symbol an und besetzt die Attribute.
	- parameters:
		- start: Startknoten
		- ende: Endknoten
		- gerichtet: wenn wahr, ist der Weg gerichtet
		- gewicht: Kantengewicht
		- breite: Breite der Linie
		- farbe: Farbe der Linie
    */
    init(start: KnotenSymbol, ende: KnotenSymbol, gerichtet: Bool, gewicht: String, breite: Int, farbe: String)
    {
        von = start
        nach = ende
        self.farbe = farbe
        self.breite = breite
        self.gerichtet = gerichtet
        self.gewicht = gewicht
        außen = Rechteck()
        innen = Rechteck()
        pfeil = Dreieck()
        pfeil.FarbeSetzen(farbe: "schwarz")
        text = Text()
        text.SichtbarkeitSetzen(sichtbar: !("" == gewicht))
        text.TextSetzen(text: gewicht)
        DarstellungAktualisieren()
    }
    
    /**
    Aktualisiert das Symbol
    */
    func DarstellungAktualisieren()
    {
        let x1 =  Double(von.XGeben())
        let y1 = Double(von.YGeben())
        let x2 = Double(nach.XGeben())
        let y2 = Double(nach.YGeben())
        let länge = Int(sqrt((x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2)))
        let xm = Int((x1 + x2) / 2.0)
        let ym = Int((y1 + y2) / 2.0)
        let x = xm - länge / 2
        let y = ym - breite / 2
        außen.PositionSetzen(x: x, y: y)
        außen.GrößeSetzen(breite: länge, höhe: breite)
        if breite < 4
        {
            let größe = breite * 4
            innen.SichtbarkeitSetzen(sichtbar: false)
            außen.FarbeSetzen(farbe: farbe)
            pfeil.SichtbarkeitSetzen(sichtbar: gerichtet)
            pfeil.PositionSetzen(x: xm, y: ym - größe / 2)
            pfeil.GrößeSetzen(breite: größe, höhe: größe)
            pfeil.WinkelSetzen(winkel: 270)
        }
        else
        {
            let delta = breite <= 8 ? 2 : 4
            innen.SichtbarkeitSetzen(sichtbar: true)
            innen.PositionSetzen(x: x, y: y + delta / 2)
            innen.GrößeSetzen(breite: länge, höhe: breite - delta)
            innen.FarbeSetzen(farbe: farbe)
            außen.FarbeSetzen(farbe: "schwarz")
            pfeil.SichtbarkeitSetzen(sichtbar: gerichtet)
            pfeil.PositionSetzen(x: xm, y: ym - breite / 2)
            pfeil.GrößeSetzen(breite: breite - delta, höhe: breite)
            pfeil.WinkelSetzen(winkel: 270)
        }
        
        var winkel = 0.0
        if x1 == x2
        {
            if y2 > y1
            {
                winkel = -90
            }
            else
            {
                winkel = 90
            }
        }
        else if x1 < x2
        {
            winkel = -atan((y2 - y1) / (x2 - x1)) / Double.pi * 180.0
        }
        else
        {
            winkel = 180 - atan((y2 - y1) / (x2 - x1)) / Double.pi * 180.0
        }
        pfeil.WinkelSetzen(winkel: Int(winkel) + 270)
        innen.WinkelSetzen(winkel: Int(winkel))
        außen.WinkelSetzen(winkel: Int(winkel))
        pfeil.GanzNachHintenBringen()
        innen.GanzNachHintenBringen()
        außen.GanzNachHintenBringen()
        if (-90 <= winkel) && (winkel <= 0)
        {
            let w = (winkel + 90) / 180.0 * Double.pi
            let delta = Double(breite)
            text.PositionSetzen(x: xm + Int(delta * cos(w)), y: ym - Int(delta * sin(w)))
        }
        else if (0 < winkel) && (winkel <= 90)
        {
            let w = (winkel + 90) / 180.0 * Double.pi
            let delta = Double(breite)
            text.PositionSetzen(x: xm - Int(delta * cos(w)), y: ym + Int(delta * sin(w)) + 7)
        }
        else if (90 < winkel) && (winkel <= 180)
        {
            let w = (winkel + 90) / 180.0 * Double.pi
            let delta = Double(breite)
            text.PositionSetzen(x: xm - Int(delta * cos(w)), y: ym + Int(delta * sin(w)))
        }
        else
        {
            let w = (winkel + 90) / 180.0 * Double.pi
            let delta = Double(breite)
            text.PositionSetzen(x: xm + Int(delta * cos(w)), y: ym - Int(delta * sin(w)) + 10)
        }
    }
    
    /**
    Entfernt das Kantensymbol aus der Anzeige
    */
    func Entfernen()
    {
        außen.Entfernen()
        innen.Entfernen()
        pfeil.Entfernen()
        text.Entfernen()
    }
    
    /**
	Setzt die Farbe der Darstellung
	- parameters:
		- f: die (neue) Farbe
    */
    func FarbeSetzen(f: String)
    {
        farbe = f
        DarstellungAktualisieren()
    }
    
    /**
	Meldet den Startknoten
	- returns: Startknoten
    */
    func StartsymbolGeben() ->KnotenSymbol
    {
        return von
    }
    
    /**
	Meldet den Zielknoten
	- returns: Zielknoten
    */
    func ZielsymbolGeben() -> KnotenSymbol
    {
        return nach
    }
}

/**
Beschreibt einen Knoten

- author: Albert Wiedemann
- version: 1.0
*/
class Knoten
{
    /** Der Bezeichner des Knotens */
    private var bezeichner: String
    /** Das Darstellungssymbol des Knotens */
    private var symbol: KnotenSymbol
    /** Die Länge des bisherigen Wegs in Kantenschritten */
    private var länge: Int

    /** Vorgängerknoten für die Union-Find-Datenstrutkur bzw. nil, falls kein Vorgänger existiert */
    private var vorgänger: Knoten?
    /** Hilfsvariable für die Preorder-Traversierung */
    private var dfsBesucht: Bool
	/** Katen zu den Nachbarknoten im minimalen Spannbaum */
    private var baumnachbarn: [Kante]

    /**
	Besetzt die Attribute und legt das Knotensymbol an.
	- parameters:
		- bezeichner: Bezeichner
		- x: x-Koordinate
		- y: y-Koordinate
    */
    init(bezeichner: String, x: Int, y: Int)
    {
        self.bezeichner = bezeichner
        symbol = KnotenSymbol(x: x, y: y, radius: 20, farbe: "weiß", bezeichner: bezeichner)
        länge = 0
        vorgänger = nil
        dfsBesucht = false
        baumnachbarn = []
    }
    
    /**
    Meldet den Bezeichner des Knotens zurück.
	- returns: Bezeichner
    */
    func BezeichnerGeben() -> String
    {
        return bezeichner
    }
    
    /**
	Meldet das Symbol des Knotens zurück.
	- returns: Knotensymbol
    */
    func SymbolGeben() -> KnotenSymbol
    {
        return symbol
    }
    
    /**
	Setzt die Farbe der Darstellung
	- parameters:
		- f: die (neue) Farbe
    */
    func FarbeSetzen(f: String)
    {
        symbol.FarbeSetzen(f: f)
    }

    /**
 	Setzt die Weglänge
    - parameters:
    	- l: die (neue) Länge
     */
    func LängeSetzen(l: Int)
    {
        länge = l
    }

    /**
	Meldet die Weglänge zurück.
	- returns: Weglänge
     */
    func LängeGeben() -> Int
    {
        return länge
    }

    /**
	Setzt den Vorgänger
	- parameters:
     	- v: der Vorgänger
     */
    func VorgängerSetzen(v: Knoten?)
    {
        vorgänger = v
    }

    /**
	Meldet den Vorgänger zurück.
	- returns: Vorgänger
     */
    func VorgängerGeben() -> Knoten?
    {
        return vorgänger
    }

    /**
    Fügt einen Nachbarknoten hinzu
    - parameters:
         - n: Kante zum Nachbarknoten
     */
    func BaumnachbarSetzen(n: Kante)
    {
        baumnachbarn.append(n)
    }

    /**
     Gibt das Feld der Kanten zu den Nachbarknoten zurück
     - returns: Kanten zu den Nachbarknoten
     */
    func BaumnachbarnGeben() -> [Kante]
    {
        return baumnachbarn
    }
    
    /**
     Meldet den Wert der Variablen dsfBesucht
     - returns dsfBesucht
     */
    func DfsBesuchtGeben() -> Bool
    {
        return dfsBesucht
    }
    
    /**
     Setzt den Wert der Variablen dsfBesucht auf wahr (besucht)
     */
    func DfsBesuchtSetzen()
    {
        dfsBesucht = true
    }
    
    /**
     Setzt den Wert der Variablen dsfBesucht auf falsch (nicht besucht)
     */
    func DfsUnbesuchtSetzen()
    {
        dfsBesucht = false
    }
}

/**
Kantenbeschreibung für die Konstruktion des minimalen Spannbaums

- author: Johannes Neumeyer
- version: 1.0
 */
class Kante
{
    /** Länge der Kante */
    private let länge: Int
    /** Startknoten der Kante */
    private let start: Knoten
    /** Zielknoten der Kante */
    private let ziel: Knoten
    /** Symbol für die Darstellung der Kante */
    private let symbol: KantenSymbol

    /**
    Besetzt die Attribute
     */
    init(start: Knoten, ziel: Knoten, länge: Int)
    {
        self.start = start
        self.ziel = ziel
        self.länge = länge
        symbol = KantenSymbol(start: start.SymbolGeben(), ende: ziel.SymbolGeben(), gerichtet: false, gewicht: "\(länge)", breite: 3, farbe: "blau")
    }
    
    /**
	Setzt die Farbe der Darstellung
	- parameters:
     	- f die (neue) Farbe
     */
    func FarbeSetzen(f: String)
    {
        symbol.FarbeSetzen(f: f)
    }

    /**
	Meldet den Startknoten
	- returns: Startknoten
     */
    func StartGeben() -> Knoten
    {
        return start
    }
    
    /**
    Meldet den Zielknoten
    - returns: Zielknoten
     */
    func ZielGeben() -> Knoten
    {
        return ziel
    }
    
    /**
    Meldet die Länge der Kante
    - returns: Länge
     */
    func LängeGeben() -> Int
    {
        return länge
    }
    
    /**
    Meldet das Kantensymbol
    - returns: Kantensymbol
     */
    func KantensymbolGeben() -> KantenSymbol
    {
        return symbol
    }
}

/**
Stellt Methoden zum Lesen der Graphendaten bereit

- author: Albert Wiedemann
- version: 1.0
*/
class Lesen
{
    /**
	Liest die Datenbank unter dem gegebenen Namen ein und trägt die Daten in den Graphen ein.
	- parameters:
		- name: Name der Datenbankcontainerdatei
		- g: das Graphenobjekt, in dem die Daten eingetragen werden sollen
    */
    func LesenDatenbank(name: String, g: Graph) -> Bool
    {
        do
        {
            let verbindung = try DatenbankVerbindung(name: name)
            var anweisung = try verbindung.AnweisungErzeugen(sql: "SELECT bezeichner, x, y FROM knoten")
            var daten = try anweisung.AbfrageAusführen()
            while daten.AufNächstenDatensatzPositionieren()
            {
                try g.KnotenEinfügen(bezeichner: daten.SpaltenwertGebenString(spaltenname: "bezeichner"), x: daten.SpaltenwertGebenInt(spaltenname: "x"), y: daten.SpaltenwertGebenInt(spaltenname: "y"))
            }
            anweisung.AnweisungSchliessen()
            anweisung = try verbindung.AnweisungErzeugen(sql: "SELECT bezeichnerStart, bezeichnerZiel, gewicht, gerichtet FROM kanten")
            daten = try anweisung.AbfrageAusführen()
            while daten.AufNächstenDatensatzPositionieren()
            {
                try g.KanteEinfügen(von: daten.SpaltenwertGebenString(spaltenname: "bezeichnerStart"), nach: daten.SpaltenwertGebenString(spaltenname: "bezeichnerZiel"), gewichtung: daten.SpaltenwertGebenInt(spaltenname: "gewicht"))
            }
            anweisung.AnweisungSchliessen()
            verbindung.VerbindungSchliessen()
            return true
        }
        catch
        {
            if let dberror = error as? SQLiteFehler
            {
            	print ("SQLite-Fehler beim Bearbeiten der Datenbank: \(dberror.localizedDescription)")
            }
            else
            {
                print("Unbekannter Fehler beim Lesen der Datenbank: \(error)")
            }
            return false;
        }
    }
    
    /**
	Liest die Datei unter dem gegebenen Namen ein und trägt die Daten in den Graphen ein.
	- parameters:
		- name: Name der Datei
		- g: das Graphenobjekt, in dem die Daten eingetragen werden sollen
    */
    func LesenDatei(name: String, g: Graph) -> Bool
    {
        do
        {
            guard let datei = Bundle.main.url(forResource: name, withExtension: "")
            else
            {
            	print ("Die Datei \(name) exisitert nicht")
            	return false
            }
            let zeilen = try String(contentsOf: datei, encoding: .utf8).split(separator: "\n")
            var anzahl: Int = Int(zeilen[0].suffix(from: zeilen[0].index(after: zeilen[0].lastIndex(of: " ")!)))!
            var pos = 1
            for _ in 1 ... anzahl
            {
            	let teile = zeilen[pos].split(separator: "\t")
                g.KnotenEinfügen(bezeichner: String(teile[2]), x: Int(teile[0])!, y: Int(teile[1])!)
            	pos += 1
            }
            anzahl = Int(zeilen[pos].suffix(from: zeilen[pos].index(after: zeilen[pos].lastIndex(of: " ")!)))!
            pos += 1
            for _ in 1 ... anzahl
            {
            	let teile = zeilen[pos].split(separator: "\t")
                g.KanteEinfügen(von: String(teile[0]), nach: String(teile[1]), gewichtung: Int(teile[2])!)
            	pos += 1
            }
            return true
        }
        catch
        {
            print("Fehler beim Lesen der Datei: \(error)")
            return false
        }
	}
}

/**
Verwaltet einen ungerichteten, gewichteten Graphen mittels Adjazenzmatrix

- author: Johannes Neumeyer, Albert Wiedemann
- version: 1.0
*/
class Graph
{
    /** Feld der Knoten des Graphen */
    private var knoten: [Knoten]
    /** 2-dim Feld der Adjazenzmatrix */
    private var matrix: [[Int]]
    /** Feld der Kantensymbole des Graphen */
    private var kanten: [Kante]
    
    /**
    Baut die Datenstruktur auf
    */
    init()
    {
        knoten = [Knoten]()
        matrix = [[Int]]()
        kanten = [Kante]()
    }
    
    /**
	Einfügen eines neuen Knoten in den Graphen
	- parameters:
		- bezeichner: Bezeichner des neuen Knotens, der dem Graphen hinzugefügt wird.
		- x: x-Koordinate für die Anzeige des Knotens
		- y: y-Koordinate für die Anzeige des Knotens
    */
    func KnotenEinfügen(bezeichner: String, x: Int, y: Int)
    {
        knoten.append(Knoten(bezeichner: bezeichner, x: x, y: y))
        for index in 0 ..< matrix.count
        {
            matrix[index].append(-1)
        }
        matrix.append([Int]())
        for _ in 0 ..< matrix.count
        {
            matrix[matrix.count - 1].append(-1)
        }
    }

    /**
	Gibt den Knoten mit dem angegebenen Bezeichner zurück.
	Wenn ein Knoten mit diesem Bezeichner nicht bekannt ist wird null zurückgegeben
	- parameters:
		- bezeichner: Bezeichner des Knoten der gesucht wird
	- returns: Referenz auf das Knotenobjekt oder null
    */
    private func KnotenGeben(bezeichner: String) -> Knoten?
    {
        for k in knoten
        {
            if k.BezeichnerGeben() == bezeichner
            {
                return k
            }
        }
        return nil
    }

    /**
	Gibt die interne Nummer des Knotens
	Wenn ein Knoten mit diesem Bezeichner nicht bekannt ist wird -1 zurückgegeben
	- parameters:
		- bezeichner: Bezeichner des Knoten der gesucht wird
	- returns: Indexnummer des Knotens im Knotenfeld; 0 <= res < knoten.count bzw. -1
    */
    private func KnotenNummerGeben(bezeichner: String) -> Int
    {
        for index in 0 ..< knoten.count
        {
            if knoten[index].BezeichnerGeben() == bezeichner
            {
                return index
            }
        }
        return -1
    }
    
    /**
	Gibt die Bezeichnung eines Knotens mit der internen Knotennummer
	- parameters:
		- knotenNummer: Indexnummer des Knotens im Knotenarray; 0 <= x < knoten.count
	- returns: Bezeichner des Knotens
    */
    func KnotenBezeichnerGeben(knotenNummer: Int) -> String
    {
        if (knotenNummer < knoten.count) && (knotenNummer >= 0)
        {
            return knoten[knotenNummer].BezeichnerGeben()
        }
        else
        {
            return ""
        }
    }

    /**
	Einfügen einer Kante in den Graphen
	Eine Kante ist durch einen Anfangsknoten und einen Endknoten festgelegt und hat eine Gewichtung
	- parameters:
		- von: Bezeichner des Anfangsknotens
		- nach: Bezeichner des Endknotens
		- gewichtung: Gewichtung der Kante als Ganzzahl
    */
    func KanteEinfügen(von: String, nach: String, gewichtung: Int)
    {
        let vonNummer = KnotenNummerGeben(bezeichner: von)
        let nachNummer = KnotenNummerGeben(bezeichner: nach)
        if (vonNummer != -1) && (nachNummer != -1) && (vonNummer != nachNummer)
        {
            matrix[vonNummer][nachNummer] = gewichtung
            matrix[nachNummer][vonNummer] = gewichtung
            let neueKante = Kante(start: knoten[vonNummer], ziel: knoten[nachNummer], länge: gewichtung)
            kanten.append(neueKante)
        }
    }

    /**
	Gibt die Adjazenzmatrix des Graphen in der Konsole aus
	Nach Zeilen und Spalten formatiert
	Als Spaltenbreite wurde hier 4 Zeichen gewählt.
    */
    func Ausgeben()
    {
        // Kopfzeile
        print("    ", separator: "", terminator: "")
        for i in 0 ..< knoten.count
        {
            print(("\(knoten[i].BezeichnerGeben())    ").prefix(4), separator: "", terminator: "")
        }
        print("")
        
        for i in 0 ..< knoten.count
        {
            print(("\(knoten[i].BezeichnerGeben())    ").prefix(4), separator: "", terminator: "")
            for j in 0 ..< knoten.count
            {
                if matrix[i][j] != -1
                {
                    print(("\(matrix[i][j])    ").prefix(4), separator: "", terminator: "")
                }
                else
                {
                    print("    ", separator: "", terminator: "")
                }
            }
            print()
        }
    }

    /**
	Gibt die Anzahl der Knoten des Graphen zurück
	- returns:  Anzahl der Knoten
    */
    func KnotenAnzahlgeben() -> Int
    {
        return knoten.count
    }
    
    /**
	Gibt die Gewichtung einer Kante zurück
	Die Kante ist durch einen Anfangsknoten und einen Endknoten festgelegt
	- parameters:
		- von: Bezeichner des Anfangsknotens
		- nach: Bezeichner des Endknotens
	- returns: Gewichtung der Kante
    */
    func KanteGewichtGeben(von: String, nach: String) -> Int
    {
        let vonNummer = KnotenNummerGeben(bezeichner: von)
        let nachNummer = KnotenNummerGeben(bezeichner: nach)
        if (vonNummer != -1) && (nachNummer != -1)
        {
            return matrix[vonNummer][nachNummer]
        }
        else
        {
            return -1
        }
    }
    
    /**
	Löscht die Kanten und Knoten des Graphen
	Die Anzeige wird auch gelöscht
    */
    func ZurückSetzen()
    {
        for k in kanten
        {
            k.KantensymbolGeben().Entfernen()
        }
        for k in knoten
        {
            k.SymbolGeben().Entfernen()
        }
        knoten.removeAll()
        kanten.removeAll()
        matrix.removeAll()
    }

    /**
    Sucht rekursiv die Wurzel des Teilbaums, zu dem der gegebene Knoten gehört.
    Wird im Rahmen des Union-Find-Algorithmus genutzt.
    - parameters:
        - knoten: Der Knoten, dessen Wurzel gesucht wird.
    - returns: Die Wurzel des zugehörigen Teilbaums.
     */
    private func Finden(knoten: Knoten) -> Knoten
    {
        if (knoten.VorgängerGeben() == nil)
        {
            return knoten
        }
        else
        {
            return Finden(knoten: knoten.VorgängerGeben()!)
        }
    }

    /**
    Vereinigt die Teilbäume, zu denen die beiden übergebenen Knoten gehören, indem der Wurzelknoten
    des zweiten Baums auf den Wurzelknoten des ersten zeigt.
    - parameters:
        - k1: ein Knoten aus dem ersten Teilbaum
        - k2: ein Knoten aus dem zweiten Teilbaum
     */
    private func Vereinigen(k1: Knoten, k2: Knoten)
    {
        let wurzelk1 = Finden(knoten: k1)
        let wurzelk2 = Finden(knoten: k2)
        wurzelk2.VorgängerSetzen(v: wurzelk1)
    }

    /**
    Erzeugt eine Kopie der zentral verwalteten Kantenliste.
    - returns: Feld aller im Graphen vorhandenen Kanten
     */
    private func MatrixZuKantenliste() -> [Kante]
    {
        return kanten
    }

    /**
    Führt eine Preorder-Traversierung durch, beginnend bei einem zufällig ausgewählten Wurzelknoten
    des minimalen Spannbaums, und gibt die besuchte Knotenfolge als Liste zurück.
    - parameters:
        - minSpannbaum: Kanten des minimalen Spannbaums
    - returns: Liste der Knoten in Preorder-Reihenfolge
     */
    private func PreOrderGeben(minSpannbaum: [Kante]) -> [Knoten]
    {
        MinSpannbaumZuBaum(minSpannbaum: minSpannbaum)

        // Wurzel zufällig definieren
        let zufallsindex = Int.random(in: 0..<minSpannbaum.count)
        let wurzel = minSpannbaum[zufallsindex].StartGeben()
        wurzel.FarbeSetzen(f: "rot")

        return PreOrderRekursiv(wurzel: wurzel, preOrderFolge: [])
    }

    /**
    Rekursive Hilfsmethode für Preorder-Traversierungen auf Bäumen.
    Fügt die Knoten in Preorder-Reihenfolge zur übergebenen Liste hinzu.

    - parameters:
        - wurzel: Der aktuelle Wurzelknoten
        - preOrderFolge: bisher ermittelte Reihenfolge der Knoten
    - returns: aktualisierte Preorder-Knotenfolge
     */
    private func PreOrderRekursiv(wurzel: Knoten, preOrderFolge: [Knoten]) -> [Knoten]
    {
        var preOrderFolgeLokal = preOrderFolge
        preOrderFolgeLokal.append(wurzel)
        wurzel.DfsBesuchtSetzen()
        let baumnachbarn = wurzel.BaumnachbarnGeben()

        for i in 0 ..< baumnachbarn.count
        {
            let k = baumnachbarn[i]
            if k.StartGeben() === wurzel && !k.ZielGeben().DfsBesuchtGeben()
            {
                preOrderFolgeLokal = PreOrderRekursiv(wurzel: k.ZielGeben(), preOrderFolge: preOrderFolgeLokal)
            }
            else if (k.ZielGeben() === wurzel) && !k.StartGeben().DfsBesuchtGeben()
            {
                preOrderFolgeLokal = PreOrderRekursiv(wurzel: k.StartGeben(), preOrderFolge: preOrderFolgeLokal)
            }
        }
        return preOrderFolgeLokal
    }

    /**
    Initialisiert die Baum-Nachbarschaften der Knoten entsprechend dem minimalen Spannbaum.
    Fügt die entsprechenden Kanten als Baumkanten zu den Knoten hinzu.
    - parameters:
        - minSpannbaum: Kanten des minimalen Spannbaums
     */
    private func MinSpannbaumZuBaum(minSpannbaum: [Kante])
    {
        // Adjazenzen aller Knoten des minimalen Spannbaums setzen
        for i in 0 ..< minSpannbaum.count
        {
            minSpannbaum[i].StartGeben().BaumnachbarSetzen(n: minSpannbaum[i])
            minSpannbaum[i].ZielGeben().BaumnachbarSetzen(n: minSpannbaum[i])
        }
    }
}


/**
Beispiele für diverse Graphen

- author: Albert Wiedemann
- version: 1.0
*/
class Beispiele
{
    var g: Graph
    var l: Lesen
    
    /**
    Legt das Graphenobjekt an.
    */
    init()
    {
        g = Graph()
        l = Lesen()
    }

    /**
    Legt den Graphen für das Beispiel im Lehrtext an.
    Am Ende Ausgabe der Adjazenzmatrix zur Kontrolle
    */
    func LehrtextgraphAnzeigen()
    {
        g.ZurückSetzen()
        if l.LesenDatenbank(name: "Handlungsreise_Lehrtext.grdb", g: g)
        {
            g.Ausgeben()
        }
    }

   /**
    Legt einen Graphen für den Deutschlandgraphen an.
    Am Ende Ausgabe der Adjazenzmatrix zur Kontrolle
    */
    func DeutschlandgraphAnzeigen()
    {
        g.ZurückSetzen()
        if l.LesenDatenbank(name: "Handlungsreise_Deutschland.grdb", g: g)
        {
            g.Ausgeben()
        }
    }

/*
    func NearestNeighbour_Lehrtext(start: String)
    {
        g.ZurückSetzen()
		if l.LesenDatenbank(name: "Handlungsreise_Lehrtext.grdb", g: g)
        {
            g.Ausgeben()
            g.NearestNeighbour(start: start)
        }
    }
*/

/*
    func NearestNeighbour_Deutschland(start: String)
    {
        g.ZurückSetzen()
		if l.LesenDatenbank(name: "Handlungsreise_Deutschland.grdb", g: g)
        {
            g.Ausgeben()
            g.NearestNeighbour(start: start)
        }
    }
*/

/*
    func Kruskal_Lehrtext()
    {
    	g.ZurückSetzen();
		if l.LesenDatenbank(name: "Handlungsreise_Lehrtext.grdb", g: g)
   		{
			g.Ausgeben()
			g.ApproximationMitKruskal()
		}
    }
*/

/*
    func Kruskal_Deutschland()
    {
    	g.ZurückSetzen();
		if l.LesenDatenbank(name: "Handlungsreise_Deutschland.grdb", g: g)
   		{
			g.Ausgeben()
			g.ApproximationMitKruskal()
		}
    }
*/
}

let b = Beispiele()
b.LehrtextgraphAnzeigen()

