import Foundation
import SQLite3

/**
Beschreibt die vertschiedenen Fehlermöglichkeiten bei der Verwendung von SQLite

- author: Albert Wiedemann
- version: 1.0
*/
public enum SQLiteFehler: Error
{
  case open(text: String)
  case prepare(text: String)
  case step(text: String)
  case bind(text: String)
  case get(text: String)
}

/**
Stellt die Verbindung zur Datenbank her

- author: Albert Wiedemann
- version: 1.0
*/
public class DatenbankVerbindung
{
    /** Verweis auf die Datenbank */
    var db: OpaquePointer?
    /** Die letzte Fehlermeldung */
    var fehlerMeldung: String
    {
        if let meldung = sqlite3_errmsg(db)
        {
            let fehlerMeldung = String(cString: meldung)
            return fehlerMeldung
        }
        else
        {
            return "Keine Fehlermeldung von SQLite."
        }
    }
    
    /**
    Löscht die Datenbank im Zielverzeichnis
    
    Wenn die Datenbank nicht vorhanden ist, wird einfach nichts gemacht
    - parameters:
        - name: Name der Datenbankdatei.
    */
    public class func DBRücksetzen(name: String)
    {
        let ziel = try? FileManager.default.url(for: .documentDirectory, in: .userDomainMask, appropriateFor: nil, create: true).appendingPathComponent(name)
        if FileManager.default.fileExists(atPath: ziel!.relativePath)
        {
            try? FileManager.default.removeItem(at: ziel!)
        }
    }
    
    /**
    Der Konstruktor stellt die Verbindung zur Datenbank her
    
    Ist das Herstellen der Verbindung nicht möglich, wird ein Ausnahmezustand ausgelöst
    - parameters:
        - name: Name der Datenbankdatei.
    */
    public init(name: String) throws
    {
        let ziel = try? FileManager.default.url(for: .documentDirectory, in: .userDomainMask, appropriateFor: nil, create: true).appendingPathComponent(name)
        do
        {
            guard let quelle = Bundle.main.url(forResource: name, withExtension: "")
            else
            {
                throw SQLiteFehler.open(text: "Datenbankdatei \(name) existiert nicht.")
            }
            if !FileManager.default.fileExists(atPath: ziel!.relativePath)
            {
                try FileManager.default.copyItem(at: quelle, to: ziel!)
            }
        }
        catch
        {
            throw SQLiteFehler.open(text: "\(error)")
        }
        let res = sqlite3_open_v2(ziel?.path, &db, SQLITE_OPEN_READWRITE, nil)
        if res != SQLITE_OK
        {
            if let meldung = sqlite3_errmsg(db)
            {
                throw SQLiteFehler.open(text: String(cString: meldung))
            }
            else
            {
                throw SQLiteFehler.open(text: "SQLite kann die Datenbank \(name) nicht öffnen.")
            }
        }
    }
    
    /**
    Der Destruktor schließt gegebenenfalls die Verbindung zur Datenbankdatei
    */
    deinit
    {
        if db != nil
        {
            sqlite3_close(db)
        }
    }
    
    /**
    Erzeugt eine neue Anweisung
    - parameters:
        - sql: Die auszuführende Anweisung
    - returns: ein Objekt zur Verwaltung der Anweisung
    */
    public func AnweisungErzeugen(sql: String) throws -> Anweisung
    {
        if db != nil
        {
            return Anweisung(sql: sql, db: db!)
        }
        else
        {
            throw SQLiteFehler.open(text: "Die Anweisung kann nicht erzeugt werden, da keine Datenbank geöffnet ist.")
        }
    }
    
    /**
    Schließt die Verbindung zur Datenbankdatei
     */
    public func VerbindungSchliessen()
    {
        if db != nil
        {
            sqlite3_close(db)
            db = nil
        }
    }
}

/**
Veraltet eine enzelne Datenbankanweisung

- author: Albert Wiedemann
- version: 1.0
*/
public class Anweisung
{
    /** Anweisungstext */
    let sqlanweisung: String
    /** Anweisung vorbereitet */
    var vorbereitet: Bool
    /** SQLite - Objekt */
    var anweisung: OpaquePointer?
    /** Datenbank */
    var db: OpaquePointer
    /** Spaltennamen */
    var namen: [String]
    /** Anzahl der Spalten */
    var spaltenAnzahl: Int
    
    /**
    Der Konstruktor speichert die SQL-Anweisung
    - parameters:
        - sql: Die auszuführende Anweisung
        - db: Referenz auf die Verbindung zur Datenbankdatei
    */
    init(sql: String, db: OpaquePointer)
    {
        sqlanweisung = sql
        vorbereitet = false
        anweisung = nil
        self.db = db
        namen = []
        spaltenAnzahl = 0
    }
    
    /**
    Der Destruktor invalidiert das Anweisungsobjekt
     */
    deinit
    {
        AnweisungSchliessen()
    }
    
    /**
    Führt die SQL-Anweisung aus
    */
    public func AnweisungAusführen() throws
    {
        if !vorbereitet
        {
            if sqlite3_prepare_v2(db, sqlanweisung, -1, &anweisung, nil) == SQLITE_OK
            {
                vorbereitet = true
            }
            else
            {
                anweisung = nil
                if let meldung = sqlite3_errmsg(db)
                {
                    throw SQLiteFehler.prepare(text: String(cString: meldung))
                }
                else
                {
                    throw SQLiteFehler.prepare(text: "SQLite kann die Anweisung \(sqlanweisung) nicht vorbereiten.")
                }
            }
        }
        if sqlite3_step(anweisung) != SQLITE_DONE
        {
            if let meldung = sqlite3_errmsg(db)
            {
                throw SQLiteFehler.step(text: String(cString: meldung))
            }
            else
            {
                throw SQLiteFehler.step(text: "SQLite kann die Anweisung \(sqlanweisung) nicht ausführen.")
            }
        }
    }
    
    /**
    Bestimmt die Spaltennummer zu einem gegebenen Spaltennamen
    - parameters:
        - name: Spaltenname
    - returns: Spaltennummer oder nil
    */
    fileprivate func SpaltenNummerGeben (name: String) -> Int?
    {
        return namen.firstIndex(where: {$0 == name})
    }
    
    /**
    Führt die Abfrage aus und liefert eine Referenz auf die Ergebnistabelle
    - returns: Ergebnistabelle
    */
    public func AbfrageAusführen() throws -> ErgebnisTabelle
    {
        if !vorbereitet
        {
            if sqlite3_prepare_v2(db, sqlanweisung, -1, &anweisung, nil) == SQLITE_OK
            {
                vorbereitet = true
            }
            else
            {
                anweisung = nil
                if let meldung = sqlite3_errmsg(db)
                {
                    throw SQLiteFehler.prepare(text: String(cString: meldung))
                }
                else
                {
                    throw SQLiteFehler.prepare(text: "SQLite kann die Anweisung \(sqlanweisung) nicht vorbereiten.")
                }
            }
        }
        let max = sqlite3_column_count(anweisung)
        spaltenAnzahl = Int(max)
        for nummer: Int32 in 0 ..< max
        {
            namen.append(String(cString: sqlite3_column_name(anweisung, nummer)))
        }
        return ErgebnisTabelle(a: anweisung, anweisung: self)
    }
    
    /**
    Schließt die Anweisung und räumt den Server auf.
    
    Die Anweisung kann später erneut ausgeführt werden.
     */
    public func AnweisungSchliessen()
    {
        if anweisung != nil
        {
            sqlite3_finalize(anweisung)
            anweisung = nil
            vorbereitet = false
        }
    }
}

/**
Verwaltet die Ergebnistabelle einer Abfrage

- author: Albert Wiedemann
- version: 1.0
*/
public class ErgebnisTabelle
{
    /** Referenz auf das Anweisungsobjekt in der Datenbankverwaltung */
    var sqliteAnweisung: OpaquePointer?
    /** Referenz auf die Anweisung */
    var anweisung: Anweisung

    /**
    Der Konstruktor legt die Attribute an
    - parameters:
        - a: Referenz auf das Anweisungsobjekt in der Datenbankverwaltung
    */
    init (a: OpaquePointer?, anweisung: Anweisung)
    {
        sqliteAnweisung = a
        self.anweisung = anweisung
    }
    
    /**
    Positioniert auf den nächsten Datensatz
    - returns: wahr, wenn der nächste Datensatz existiert
    */
    public func AufNächstenDatensatzPositionieren() -> Bool
    {
        return sqlite3_step(sqliteAnweisung) == SQLITE_ROW
    }
    
    /**
    Setzt die Position in der Ergebnistabelle wieder vor den ersten Datensatz
    */
    public func Zurücksetzen()
    {
        sqlite3_reset(sqliteAnweisung)
    }
    
    /**
    Gibt den Wert der gegebenen Spalte als ganze Zahl zurück.
    - parameters:
        - spaltennummer: Die Nummer der Spalte (Zählung ab 0)
    - returns: Wert der Spalte als ganze Zahl
    */
    public func SpaltenwertGebenInt (spaltennummer: Int) -> Int
    {
        return Int(sqlite3_column_int(sqliteAnweisung, Int32(spaltennummer)))
    }
    
    /**
    Gibt den Wert der gegebenen Spalte als ganze Zahl zurück.
    
    - parameters:
        - spaltenname: Der Name der Spalte
    - returns: Wert der Spalte als ganze Zahl
    */
    public func SpaltenwertGebenInt (spaltenname: String) throws -> Int
    {
        guard let spaltennummer = anweisung.SpaltenNummerGeben(name: spaltenname)
        else
        {
            throw SQLiteFehler.get(text: "Spaltenname \(spaltenname) existiert nicht")
        }
        return Int(sqlite3_column_int(sqliteAnweisung, Int32(spaltennummer)))
    }
    
    /**
    Gibt den Wert der gegebenen Spalte als Zeichenkette zurück.
    
    - parameters:
        - spaltennummer: Die Nummer der Spalte (Zählung ab 0)
    - returns: Wert der Spalte als Zeichenkette
    */
    public func SpaltenwertGebenString (spaltennummer: Int) -> String
    {
        if let cStr = sqlite3_column_text(sqliteAnweisung, Int32(spaltennummer))
        {
            return String(cString: cStr)
        }
        else
        {
            return ""
        }
    }
    
    /**
    Gibt den Wert der gegebenen Spalte als Zeichenkette zurück.
    
    - parameters:
        - spaltenname: Der Name der Spalte
    - returns: Wert der Spalte als Zeichenkette
    */
    public func SpaltenwertGebenString (spaltenname: String) throws -> String
    {
        guard let spaltennummer = anweisung.SpaltenNummerGeben(name: spaltenname)
        else
        {
            throw SQLiteFehler.get(text: "Spaltenname \(spaltenname) existiert nicht")
        }
        if let cStr = sqlite3_column_text(sqliteAnweisung, Int32(spaltennummer))
        {
            return String(cString: cStr)
        }
        else
        {
            return ""
        }
    }
    
    /**
    Gibt den Wert der gegebenen Spalte als Kommazahl zurück.
    
    - parameters:
        - spaltennummer: Die Nummer der Spalte (Zählung ab 0)
    - returns: Wert der Spalte als Kommazahl
    */
    public func SpaltenwertGebenDouble (spaltennummer: Int) -> Double
    {
        return sqlite3_column_double(sqliteAnweisung, Int32(spaltennummer))
    }
    
    /**
    Gibt den Wert der gegebenen Spalte als Kommazahl zurück.
    
    - parameters:
        - spaltenname: Der Name der Spalte
    - returns: Wert der Spalte als Kommazahl
    */
    public func SpaltenwertGebenDouble (spaltenname: String) throws -> Double
    {
        guard let spaltennummer = anweisung.SpaltenNummerGeben(name: spaltenname)
        else
        {
            throw SQLiteFehler.get(text: "Spaltenname \(spaltenname) existiert nicht")
        }
        return sqlite3_column_double(sqliteAnweisung, Int32(spaltennummer))
    }
}



